# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import fields, models, api


class LoyaltyCard(models.Model):
    _name = 'loyalty.card'
    _inherit = ['loyalty.card', 'pos.load.mixin']

    source_pos_order_id = fields.Many2one('pos.order', "PoS Order Reference",
        help="PoS order where this coupon was generated.")
    source_pos_order_partner_id = fields.Many2one(
        'res.partner', "PoS Order Customer",
        related="source_pos_order_id.partner_id")

    @api.model
    def _load_pos_data_domain(self, data, config):
        return False

    @api.model
    def _load_pos_data_fields(self, config):
        return ['partner_id', 'code', 'points', 'program_id', 'expiration_date', 'write_date']

    def _has_source_order(self):
        return super()._has_source_order() or bool(self.source_pos_order_id)

    def _get_default_template(self):
        self.ensure_one()
        if self.source_pos_order_id:
            return self.env.ref('pos_loyalty.mail_coupon_template', False)
        return super()._get_default_template()

    def _mail_get_partner_fields(self, introspect_fields=False):
        return super()._mail_get_partner_fields(introspect_fields=introspect_fields) + ['source_pos_order_partner_id']

    def _get_signature(self):
        return self.source_pos_order_id.user_id.signature or super()._get_signature()

    def _compute_use_count(self):
        super()._compute_use_count()
        read_group_res = self.env['pos.order.line']._read_group(
            [('coupon_id', 'in', self.ids)], ['coupon_id'], ['__count'])
        count_per_coupon = {coupon.id: count for coupon, count in read_group_res}
        for card in self:
            card.use_count += count_per_coupon.get(card.id, 0)

    @api.model
    def get_gift_card_status(self, gift_code, config_id):
        card = self.search([('code', '=', gift_code)], limit=1)
        is_valid = card.exists() and (not card.expiration_date or card.expiration_date > fields.Date.today()) and card.points > 0
        is_valid = is_valid and (card.program_id.program_type == 'gift_card') and not card.partner_id
        is_valid = is_valid and len([id for id in card.history_ids.mapped('order_id') if id != 0]) == 0
        card_fields = self._load_pos_data_fields(config_id)
        return {
            'status': bool(is_valid) or not card.exists(),
            'data': {
                'loyalty.card': card.read(card_fields, load=False),
            }
        }

    @api.model
    def get_loyalty_card_partner_by_code(self, code):
        return self.env['loyalty.card'].search([
            ('code', '=', code),
            ('program_type', '=', 'loyalty'),
        ], limit=1).partner_id or False

    def _send_creation_communication(self, force_send=False):
        """
        Override to log gift card emails in pos.order chatter
        """
        mail_ids = super()._send_creation_communication(force_send=force_send)
        for mail in self.env['mail.mail'].browse([mid for mid in mail_ids if mid]):
            if mail.model != 'loyalty.card' or mail.res_id not in self.ids:
                continue

            card = self.browse(mail.res_id)
            if card.program_id.program_type == 'gift_card' and card.source_pos_order_id:
                card.source_pos_order_id.message_post(body=mail.body_content)
        return mail_ids
