import { after, beforeEach, expect, getFixture, resize, test } from "@odoo/hoot";
import {
    click,
    drag,
    edit,
    hover,
    keyDown,
    keyUp,
    leave,
    pointerDown,
    press,
    queryAll,
    queryAllTexts,
    queryFirst,
    queryOne,
    queryRect,
    queryText,
    setInputFiles,
} from "@odoo/hoot-dom";
import {
    Deferred,
    advanceFrame,
    advanceTime,
    animationFrame,
    disableAnimations,
    mockTouch,
    runAllTimers,
    tick,
} from "@odoo/hoot-mock";
import { Component, onRendered, onWillRender, xml } from "@odoo/owl";
import {
    MockServer,
    clickKanbanLoadMore,
    contains,
    createKanbanRecord,
    defineActions,
    defineModels,
    defineParams,
    discardKanbanRecord,
    editKanbanColumnName,
    editKanbanRecordQuickCreateInput,
    fields,
    getDropdownMenu,
    getFacetTexts,
    getKanbanColumn,
    getKanbanColumnDropdownMenu,
    getKanbanRecord,
    getKanbanRecordTexts,
    getPagerLimit,
    getPagerValue,
    getService,
    makeServerError,
    mockService,
    mockOffline,
    models,
    mountView,
    mountWithCleanup,
    onRpc,
    pagerNext,
    pagerPrevious,
    patchWithCleanup,
    quickCreateKanbanColumn,
    quickCreateKanbanRecord,
    removeFacet,
    serverState,
    stepAllNetworkCalls,
    switchView,
    toggleKanbanColumnActions,
    toggleKanbanRecordDropdown,
    toggleMenuItem,
    toggleMenuItemOption,
    toggleSearchBarMenu,
    validateKanbanColumn,
    validateKanbanRecord,
    validateSearch,
    webModels,
} from "@web/../tests/web_test_helpers";
import { addNewRule } from "@web/../tests/core/tree_editor/condition_tree_editor_test_helpers";

import { FileInput } from "@web/core/file_input/file_input";
import { browser } from "@web/core/browser/browser";
import { currencies } from "@web/core/currency";
import { registry } from "@web/core/registry";
import { user } from "@web/core/user";
import { RelationalModel } from "@web/model/relational_model/relational_model";
import { SampleServer } from "@web/model/sample_server";
import { KanbanCompiler } from "@web/views/kanban/kanban_compiler";
import { KanbanController } from "@web/views/kanban/kanban_controller";
import { KanbanRecord } from "@web/views/kanban/kanban_record";
import { KanbanRenderer } from "@web/views/kanban/kanban_renderer";
import { kanbanView } from "@web/views/kanban/kanban_view";
import { ViewButton } from "@web/views/view_button/view_button";
import { AnimatedNumber } from "@web/views/view_components/animated_number";
import { TOUCH_SELECTION_THRESHOLD } from "@web/views/utils";
import { WebClient } from "@web/webclient/webclient";

const { IrAttachment } = webModels;

const fieldRegistry = registry.category("fields");
const viewRegistry = registry.category("views");
const viewWidgetRegistry = registry.category("view_widgets");

async function createFileInput({ mockPost, mockAdd, props }) {
    mockService("notification", {
        add: mockAdd || (() => {}),
    });
    mockService("http", {
        post: mockPost || (() => {}),
    });
    await mountWithCleanup(FileInput, { props });
}

class Partner extends models.Model {
    _name = "partner";
    _rec_name = "foo";

    foo = fields.Char();
    bar = fields.Boolean();
    sequence = fields.Integer();
    int_field = fields.Integer({ aggregator: "sum", sortable: true });
    float_field = fields.Float({ aggregator: "sum" });
    product_id = fields.Many2one({ relation: "product" });
    category_ids = fields.Many2many({ relation: "category" });
    date = fields.Date();
    datetime = fields.Datetime();
    state = fields.Selection({
        type: "selection",
        selection: [
            ["abc", "ABC"],
            ["def", "DEF"],
            ["ghi", "GHI"],
        ],
    });
    salary = fields.Monetary({ aggregator: "sum", currency_field: "currency_id" });
    currency_id = fields.Many2one({ relation: "res.currency" });

    _records = [
        {
            id: 1,
            foo: "yop",
            bar: true,
            int_field: 10,
            float_field: 0.4,
            product_id: 3,
            category_ids: [],
            state: "abc",
            salary: 1750,
            currency_id: 1,
        },
        {
            id: 2,
            foo: "blip",
            bar: true,
            int_field: 9,
            float_field: 13,
            product_id: 5,
            category_ids: [6],
            state: "def",
            salary: 1500,
            currency_id: 1,
        },
        {
            id: 3,
            foo: "gnap",
            bar: true,
            int_field: 17,
            float_field: -3,
            product_id: 3,
            category_ids: [7],
            state: "ghi",
            salary: 2000,
            currency_id: 2,
        },
        {
            id: 4,
            foo: "blip",
            bar: false,
            int_field: -4,
            float_field: 9,
            product_id: 5,
            category_ids: [],
            state: "ghi",
            salary: 2222,
            currency_id: 1,
        },
    ];
}

class Product extends models.Model {
    _name = "product";

    name = fields.Char();
    fold = fields.Boolean({ default: false });

    _records = [
        { id: 3, name: "hello" },
        { id: 5, name: "xmo" },
    ];
}

class Category extends models.Model {
    _name = "category";

    name = fields.Char();
    color = fields.Integer();

    _records = [
        { id: 6, name: "gold", color: 2 },
        { id: 7, name: "silver", color: 5 },
    ];
}

class Currency extends models.Model {
    _name = "res.currency";

    name = fields.Char();
    symbol = fields.Char();
    position = fields.Selection({
        selection: [
            ["after", "A"],
            ["before", "B"],
        ],
    });
    inverse_rate = fields.Float();

    _records = [
        { id: 1, name: "USD", symbol: "$", position: "before", inverse_rate: 1 },
        { id: 2, name: "EUR", symbol: "€", position: "after", inverse_rate: 0.5 },
    ];
}

class User extends models.Model {
    _name = "res.users";
    has_group() {
        return true;
    }
}

defineModels([Partner, Product, Category, Currency, IrAttachment, User]);

beforeEach(() => {
    patchWithCleanup(AnimatedNumber, { enableAnimations: false });
});

test("basic ungrouped rendering", async () => {
    onRpc("web_search_read", ({ kwargs }) => {
        expect(kwargs.context.bin_size).toBe(true);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
            <templates>
                <t t-name="card">
                    <field name="foo"/>
                </t>
            </templates>
        </kanban>`,
    });

    expect(".o_kanban_view").toHaveClass("o_kanban_test");
    expect(".o_kanban_renderer").toHaveClass("o_kanban_ungrouped");
    expect(".o_control_panel_main_buttons button.o-kanban-button-new").toHaveCount(1);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    expect(".o_kanban_ghost").toHaveCount(6);
    expect(".o_kanban_record:contains(gnap)").toHaveCount(1);
});

test("kanban rendering with class and style attributes", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="myCustomClass" style="border: 1px solid red;">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect("[style*='border: 1px solid red;']").toHaveCount(0, {
        message: "style attribute should not be copied",
    });
    expect(".o_view_controller.o_kanban_view.myCustomClass").toHaveCount(1, {
        message: "class attribute should be passed to the view controller",
    });
    expect(".myCustomClass").toHaveCount(1, {
        message: "class attribute should ONLY be passed to the view controller",
    });
});

test("generic tags are case insensitive", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <Div class="test">Hello</Div>
                    </t>
                </templates>
            </kanban>`,
    });

    expect("div.test").toHaveCount(4);
});

test("kanban records are clickable by default", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect(resId).toBe(1, { message: "should trigger an event to open the form view" });
        },
    });

    await contains(".o_kanban_record").click();
});

test("kanban records with global_click='0'", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban can_open="0">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect.step("select record");
        },
    });

    await contains(".o_kanban_record").click();
    expect.verifySteps([]);
});

test("float fields are formatted properly without using a widget", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="float_field" digits="[0,5]"/>
                        <field name="float_field" digits="[0,3]"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:first").toHaveText("0.40000\n0.400");
});

test("field with widget and attributes in kanban", async () => {
    const myField = {
        component: class MyField extends Component {
            static template = xml`<span/>`;
            static props = ["*"];
            setup() {
                if (this.props.record.resId === 1) {
                    expect(this.props.attrs).toEqual({
                        name: "int_field",
                        widget: "my_field",
                        str: "some string",
                        bool: "true",
                        num: "4.5",
                        field_id: "int_field_0",
                    });
                }
            }
        },
        extractProps: ({ attrs }) => ({ attrs }),
    };
    fieldRegistry.add("my_field", myField);
    after(() => fieldRegistry.remove("my_field"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="foo"/>
                <templates>
                    <t t-name="card">
                        <field name="int_field" widget="my_field"
                            str="some string"
                            bool="true"
                            num="4.5"
                        />
                    </t>
                </templates>
            </kanban>`,
    });
});

test("kanban with integer field with human_readable option", async () => {
    Partner._records[0].int_field = 5 * 1000 * 1000;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="int_field" options="{'human_readable': true}"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual(["5M", "9", "17", "-4"]);
    expect(".o_field_widget").toHaveCount(0);
});

test.tags("desktop");
test("Hide tooltip when user click inside a kanban headers item", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <field name="product_id" options='{"group_by_tooltip": {"name": "Name"}}'/>
                <templates>
                    <t t-name="card"/>
                </templates>
            </kanban>`,
    });
    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_column_title").toHaveCount(2);
    expect(".o-tooltip").toHaveCount(0);

    await hover(".o_kanban_group:first-child .o_kanban_header_title .o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(1);

    await contains(
        ".o_kanban_group:first-child .o_kanban_header_title .o_kanban_quick_add"
    ).click();
    expect(".o-tooltip").toHaveCount(0);

    await hover(".o_kanban_group:first-child .o_kanban_header_title .o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(1);

    await contains(".o_kanban_group:first-child .o_kanban_header_title .fa-gear", {
        visible: false,
    }).click();
    expect(".o-tooltip").toHaveCount(0);
});

test.tags("desktop");
test("basic grouped rendering", async () => {
    expect.assertions(14);

    patchWithCleanup(KanbanRenderer.prototype, {
        setup() {
            super.setup(...arguments);
            onRendered(() => {
                expect.step("rendered");
            });
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });

    expect(".o_kanban_view").toHaveClass("o_kanban_test");
    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_control_panel_main_buttons button.o-kanban-button-new").toHaveCount(1);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect.verifySteps(["rendered"]);

    await toggleKanbanColumnActions(0);

    // check available actions in kanban header's config dropdown
    expect(".o-dropdown--menu .o_kanban_toggle_fold").toHaveCount(1);
    expect(".o_kanban_header:first-child .o_group_config .o_group_edit").toHaveCount(0);
    expect(".o_kanban_header:first-child .o_group_config .o_group_delete").toHaveCount(0);
    expect(".o_kanban_header:first-child .o_group_config .o_column_archive_records").toHaveCount(0);
    expect(".o_kanban_header:first-child .o_group_config .o_column_unarchive_records").toHaveCount(
        0
    );

    // focuses the search bar and closes the dropdown
    await click(".o_searchview input");

    // the next line makes sure that reload works properly.  It looks useless,
    // but it actually test that a grouped local record can be reloaded without
    // changing its result.
    await validateSearch();
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect.verifySteps(["rendered"]);
});

test("basic grouped rendering with no record", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });
    expect(".o_kanban_grouped").toHaveCount(1);
    expect(".o_view_nocontent").toHaveCount(1);
    expect(".o_control_panel_main_buttons button.o-kanban-button-new").toHaveCount(1, {
        message:
            "There should be a 'New' button even though there is no column when groupby is not a many2one",
    });
});

test.tags("desktop");
test("empty group when grouped by date", async () => {
    Partner._records[0].date = "2017-01-08";
    Partner._records[1].date = "2017-02-09";
    Partner._records[2].date = "2017-02-08";
    Partner._records[3].date = "2017-02-10";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `<kanban>
            <templates>
                <t t-name="card">
                    <field name="foo"/>
                </t>
            </templates>
        </kanban>`,
        groupBy: ["date:month"],
    });

    expect(queryAllTexts(".o_kanban_header")).toEqual(["January 2017\n(1)", "February 2017\n(3)"]);

    MockServer.env["partner"].shift(); // remove only record of the first group

    await press("Enter"); // reload
    await animationFrame();

    expect(queryAllTexts(".o_kanban_header")).toEqual(["January 2017\n(0)", "February 2017\n(3)"]);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(0);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(3);
});

test("grouped rendering with active field (archivable false)", async () => {
    // add active field on partner model and make all records active
    Partner._fields.active = fields.Boolean({ default: true });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban archivable="false">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });

    await toggleKanbanColumnActions(0);

    // check archive/restore all actions in kanban header's config dropdown
    expect(".o_column_archive_records").toHaveCount(0, { root: getKanbanColumnDropdownMenu(0) });
    expect(".o_column_unarchive_records").toHaveCount(0, { root: getKanbanColumnDropdownMenu(0) });
});

test.tags("desktop");
test("m2m grouped rendering with active field (archivable true)", async () => {
    // add active field on partner model and make all records active
    Partner._fields.active = fields.Boolean({ default: true });

    // more many2many data
    Partner._records[0].category_ids = [6, 7];
    Partner._records[3].foo = "blork";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban archivable="true">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["category_ids"],
    });

    expect(".o_kanban_group").toHaveCount(3);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(2) })).toHaveCount(2);

    expect(queryAllTexts(".o_kanban_group")).toEqual([
        "None\n(1)",
        "gold\n(2)\nyop\nblip",
        "silver\n(2)\nyop\ngnap",
    ]);

    await click(getKanbanColumn(0));
    await animationFrame();
    await toggleKanbanColumnActions(0);

    // check archive/restore all actions in kanban header's config dropdown
    // despite the fact that the kanban view is configured to be archivable,
    // the actions should not be there as it is grouped by an m2m field.
    expect(".o_column_archive_records").toHaveCount(0, { root: getKanbanColumnDropdownMenu(0) });
    expect(".o_column_unarchive_records").toHaveCount(0, { root: getKanbanColumnDropdownMenu(0) });
});

test("kanban grouped by date field", async () => {
    Partner._records[0].date = "2007-06-10";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["date"],
    });

    expect(queryAllTexts(".o_column_title")).toEqual(["None\n(3)", "June 2007\n(1)"]);
});

test("context can be used in kanban template", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field t-if="context.some_key" name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        context: { some_key: 1 },
        domain: [["id", "=", 1]],
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect(".o_kanban_record span:contains(yop)").toHaveCount(1);
});

test("kanban with sub-template", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <t t-call="another-template"/>
                    </t>
                    <t t-name="another-template">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "yop",
        "blip",
        "gnap",
        "blip",
    ]);
});

test("kanban with t-set outside card", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="int_field"/>
                <templates>
                    <t t-name="card">
                        <t t-set="x" t-value="record.int_field.value"/>
                        <div>
                            <t t-esc="x"/>
                        </div>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual(["10", "9", "17", "-4"]);
});

test("kanban with t-if/t-else on field", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field t-if="record.int_field.value > -1" name="int_field"/>
                        <t t-else="">Negative value</t>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "10",
        "9",
        "17",
        "Negative value",
    ]);
});

test("kanban with t-if/t-else on field with widget", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field t-if="record.int_field.value > -1" name="int_field" widget="integer"/>
                        <t t-else="">Negative value</t>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "10",
        "9",
        "17",
        "Negative value",
    ]);
});

test("field with widget and dynamic attributes in kanban", async () => {
    const myField = {
        component: class MyField extends Component {
            static template = xml`<span/>`;
            static props = ["*"];
        },
        extractProps: ({ attrs }) => {
            expect.step(
                `${attrs["dyn-bool"]}/${attrs["interp-str"]}/${attrs["interp-str2"]}/${attrs["interp-str3"]}`
            );
        },
    };
    fieldRegistry.add("my_field", myField);
    after(() => fieldRegistry.remove("my_field"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="foo"/>
                <templates>
                    <t t-name="card">
                        <field name="int_field" widget="my_field"
                            t-att-dyn-bool="record.foo.value.length > 3"
                            t-attf-interp-str="hello {{record.foo.value}}"
                            t-attf-interp-str2="hello #{record.foo.value} !"
                            t-attf-interp-str3="hello {{record.foo.value}} }}"
                        />
                    </t>
                </templates>
            </kanban>`,
    });
    expect.verifySteps([
        "false/hello yop/hello yop !/hello yop }}",
        "true/hello blip/hello blip !/hello blip }}",
        "true/hello gnap/hello gnap !/hello gnap }}",
        "true/hello blip/hello blip !/hello blip }}",
    ]);
});

test("view button and string interpolated attribute in kanban", async () => {
    patchWithCleanup(ViewButton.prototype, {
        setup() {
            super.setup();
            expect.step(`[${this.props.clickParams["name"]}] className: '${this.props.className}'`);
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="foo"/>
                <templates>
                    <t t-name="card">
                        <a name="one" type="object" class="hola"/>
                        <a name="two" type="object" class="hola" t-attf-class="hello"/>
                        <a name="sri" type="object" class="hola" t-attf-class="{{record.foo.value}}"/>
                        <a name="foa" type="object" class="hola" t-attf-class="{{record.foo.value}} olleh"/>
                        <a name="fye" type="object" class="hola" t-attf-class="hello {{record.foo.value}}"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect.verifySteps([
        "[one] className: 'hola oe_kanban_action'",
        "[two] className: 'hola oe_kanban_action hello'",
        "[sri] className: 'hola oe_kanban_action yop'",
        "[foa] className: 'hola oe_kanban_action yop olleh'",
        "[fye] className: 'hola oe_kanban_action hello yop'",
        "[one] className: 'hola oe_kanban_action'",
        "[two] className: 'hola oe_kanban_action hello'",
        "[sri] className: 'hola oe_kanban_action blip'",
        "[foa] className: 'hola oe_kanban_action blip olleh'",
        "[fye] className: 'hola oe_kanban_action hello blip'",
        "[one] className: 'hola oe_kanban_action'",
        "[two] className: 'hola oe_kanban_action hello'",
        "[sri] className: 'hola oe_kanban_action gnap'",
        "[foa] className: 'hola oe_kanban_action gnap olleh'",
        "[fye] className: 'hola oe_kanban_action hello gnap'",
        "[one] className: 'hola oe_kanban_action'",
        "[two] className: 'hola oe_kanban_action hello'",
        "[sri] className: 'hola oe_kanban_action blip'",
        "[foa] className: 'hola oe_kanban_action blip olleh'",
        "[fye] className: 'hola oe_kanban_action hello blip'",
    ]);
});

test("pager should be hidden in grouped mode", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });

    expect(".o_pager").toHaveCount(0);
});

test("there should be no limit on the number of fetched groups", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_GROUP_LIMIT: 1 });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2);
});

test("pager, ungrouped, with default limit", async () => {
    expect.assertions(2);

    onRpc("web_search_read", ({ kwargs }) => {
        expect(kwargs.limit).toBe(40);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_pager").toHaveCount(1);
});

test.tags("desktop");
test("pager, ungrouped, with default limit on desktop", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getPagerValue()).toEqual([1, 4]);
});

test("pager, ungrouped, with limit given in options", async () => {
    expect.assertions(1);

    onRpc("web_search_read", ({ kwargs }) => {
        expect(kwargs.limit).toBe(2);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        limit: 2,
    });
});

test.tags("desktop");
test("pager, ungrouped, with limit given in options on desktop", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        limit: 2,
    });
    expect(getPagerValue()).toEqual([1, 2]);
    expect(getPagerLimit()).toBe(4);
});

test("pager, ungrouped, with limit set on arch and given in options", async () => {
    expect.assertions(1);

    onRpc("web_search_read", ({ kwargs }) => {
        expect(kwargs.limit).toBe(3);
    });

    // the limit given in the arch should take the priority over the one given in options
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        limit: 2,
    });
});

test.tags("desktop");
test("pager, ungrouped, with limit set on arch and given in options on desktop", async () => {
    // the limit given in the arch should take the priority over the one given in options
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        limit: 2,
    });

    expect(getPagerValue()).toEqual([1, 3]);
    expect(getPagerLimit()).toBe(4);
});

test.tags("desktop");
test("pager, ungrouped, with count limit reached", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains(".o_pager_limit").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("4");
    expect.verifySteps(["search_count"]);
});

test("pager, ungrouped, with count limit reached, click next", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains(".o_pager_next").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect.verifySteps(["web_search_read"]);
});

test.tags("desktop");
test("pager, ungrouped, with count limit reached, click next on desktop", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");

    await contains(".o_pager_next").click();

    expect(".o_pager_value").toHaveText("3-4");
    expect(".o_pager_limit").toHaveText("4");
});

test("pager, ungrouped, with count limit reached, click next (2)", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    Partner._records.push({ id: 5, foo: "xxx" });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains(".o_pager_next").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect.verifySteps(["web_search_read"]);

    await contains(".o_pager_next").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect.verifySteps(["web_search_read"]);
});

test.tags("desktop");
test("pager, ungrouped, with count limit reached, click next (2) on desktop", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    Partner._records.push({ id: 5, foo: "xxx" });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");

    await contains(".o_pager_next").click();

    expect(".o_pager_value").toHaveText("3-4");
    expect(".o_pager_limit").toHaveText("4+");

    await contains(".o_pager_next").click();

    expect(".o_pager_value").toHaveText("5-5");
    expect(".o_pager_limit").toHaveText("5");
});

test("pager, ungrouped, with count limit reached, click previous", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    Partner._records.push({ id: 5, foo: "xxx" });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains(".o_pager_previous").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect.verifySteps(["search_count", "web_search_read"]);
});

test.tags("desktop");
test("pager, ungrouped, with count limit reached, click previous on desktop", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    Partner._records.push({ id: 5, foo: "xxx" });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");

    await contains(".o_pager_previous").click();

    expect(".o_pager_value").toHaveText("5-5");
    expect(".o_pager_limit").toHaveText("5");
});

test.tags("desktop");
test("pager, ungrouped, with count limit reached, edit pager", async () => {
    patchWithCleanup(RelationalModel, { DEFAULT_COUNT_LIMIT: 3 });

    Partner._records.push({ id: 5, foo: "xxx" });
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains("span.o_pager_value").click();
    // FIXME: we have to click out instead of confirming, because somehow if the
    // web_search_read calls come back too fast when pressing "Enter", another
    // RPC is triggered right after.
    await contains("input.o_pager_value").edit("2-4", { confirm: "blur" });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(3);
    expect(".o_pager_value").toHaveText("2-4");
    expect(".o_pager_limit").toHaveText("4+");
    expect.verifySteps(["web_search_read"]);

    await contains("span.o_pager_value").click();
    await contains("input.o_pager_value").edit("2-14", { confirm: "blur" });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    expect(".o_pager_value").toHaveText("2-5");
    expect(".o_pager_limit").toHaveText("5");
    expect.verifySteps(["web_search_read"]);
});

test.tags("desktop");
test("count_limit attrs set in arch", async () => {
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="2" count_limit="3">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("3+");
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    await contains(".o_pager_limit").click();

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);
    expect(".o_pager_value").toHaveText("1-2");
    expect(".o_pager_limit").toHaveText("4");
    expect.verifySteps(["search_count"]);
});

test.tags("desktop");
test("pager, ungrouped, deleting all records from last page", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <a role="menuitem" type="delete" class="dropdown-item">Delete</a>
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getPagerValue()).toEqual([1, 3]);
    expect(getPagerLimit()).toBe(4);

    // move to next page
    await pagerNext();

    expect(getPagerValue()).toEqual([4, 4]);

    // delete a record
    await contains(".o_kanban_record a").click();

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-danger").click();

    expect(getPagerValue()).toEqual([1, 3]);
    expect(getPagerLimit()).toBe(3);
});

test.tags("desktop");
test("pager, update calls onUpdatedPager", async () => {
    class TestKanbanController extends KanbanController {
        setup() {
            super.setup();
            onWillRender(() => {
                expect.step("render");
            });
        }

        async onUpdatedPager() {
            expect.step("onUpdatedPager");
        }
    }

    viewRegistry.add("test_kanban_view", {
        ...kanbanView,
        Controller: TestKanbanController,
    });
    after(() => viewRegistry.remove("test_kanban_view"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban js_class="test_kanban_view">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        limit: 3,
    });

    expect(getPagerValue()).toEqual([1, 3]);
    expect(getPagerLimit()).toBe(4);
    expect.step("next page");
    await contains(".o_pager_next").click();
    expect(getPagerValue()).toEqual([4, 4]);
    expect.verifySteps(["render", "render", "next page", "render", "onUpdatedPager"]);
});

test("click on a button type='delete' to delete a record in a column", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <a role="menuitem" type="delete" class="dropdown-item o_delete">Delete</a>
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(queryAll(".o_kanban_load_more", { root: getKanbanColumn(0) })).toHaveCount(0);

    await click(queryFirst(".o_kanban_record .o_delete", { root: getKanbanColumn(0) }));
    await animationFrame();
    expect(".modal").toHaveCount(1);

    await contains(".modal .btn-danger").click();

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(1);
    expect(queryAll(".o_kanban_load_more", { root: getKanbanColumn(0) })).toHaveCount(0);
});

test("click on a button type='archive' to archive a record in a column", async () => {
    onRpc("action_archive", ({ args }) => {
        expect.step(`archive:${args[0]}`);
        return true;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <a role="menuitem" type="archive" class="dropdown-item o_archive">archive</a>
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);

    await contains(".o_kanban_record .o_archive").click();

    expect(".modal").toHaveCount(1);
    expect.verifySteps([]);

    await contains(".modal .btn-primary").click();

    expect.verifySteps(["archive:1"]);
});

test("click on a button type='unarchive' to unarchive a record in a column", async () => {
    onRpc("action_unarchive", ({ args }) => {
        expect.step(`unarchive:${args[0]}`);
        return true;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="3">
                <templates>
                    <t t-name="card">
                        <a role="menuitem" type="unarchive" class="dropdown-item o_unarchive">unarchive</a>
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);

    await contains(".o_kanban_record .o_unarchive").click();

    expect.verifySteps(["unarchive:1"]);
});

test.tags("desktop");
test("kanban with an action id as on_create attrs", async () => {
    mockService("action", {
        doAction(action, options) {
            // simplified flow in this test: simulate a target new action which
            // creates a record and closes itself
            expect.step(`doAction ${action}`);
            MockServer.env["partner"].create({ foo: "new" });
            options.onClose();
        },
    });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban on_create="some.action">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    await createKanbanRecord();
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(5);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
        "doAction some.action",
        "web_search_read",
    ]);
});

test("Open new card in form view, without reloading the kanban view", async () => {
    defineActions([
        {
            id: 1,
            res_model: "partner",
            type: "ir.actions.act_window",
            views: [[false, "kanban"]],
        },
        {
            id: 2,
            xml_id: "some.action",
            res_model: "partner",
            type: "ir.actions.act_window",
            target: "new",
            views: [["create_view_ref", "form"]],
        },
    ]);
    Partner._views = {
        kanban: `
            <kanban on_create="some.action">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        form: `
            <form>
                <field name="foo"/>
            </form>`,
        "form,create_view_ref": `
            <form>
                <field name="foo"/>
                <footer>
                    <button string="Create Card" name="open_new_card" type="object" class="btn-primary"/>
                </footer>
            </form>`,
        search: `<search />`,
    };
    onRpc("/web/dataset/call_button/partner/open_new_card", () => {
        const newId = MockServer.env["partner"].create({ foo: "new" });
        return {
            type: "ir.actions.act_window",
            name: "Open Card",
            target: "current",
            res_model: "partner",
            res_id: newId,
            view_mode: "form",
            views: [[false, "form"]],
        };
    });

    stepAllNetworkCalls();

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    await createKanbanRecord();
    expect(`.modal`).toHaveCount(1);
    await contains(`.modal-footer button.btn-primary`).click();
    expect(`.modal`).toHaveCount(0);
    expect(".o_form_view").toHaveCount(1);
    // should not reload the first kanban view
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "/web/action/load",
        "get_views",
        "web_search_read",
        "has_group",
        "/web/action/load",
        "get_views",
        "onchange",
        "web_save",
        "open_new_card",
        "get_views",
        "web_read",
    ]);
});

test("prevent deletion when grouped by many2many field", async () => {
    Partner._records[0].category_ids = [6, 7];
    Partner._records[3].category_ids = [7];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <t t-if="widget.deletable"><span class="thisisdeletable">delete</span></t>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_foo" domain="[]" string="GroupBy Foo" context="{ 'group_by': 'foo' }"/>
            </search>`,
        groupBy: ["category_ids"],
    });

    expect(".thisisdeletable").toHaveCount(0, { message: "records should not be deletable" });

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Foo");

    expect(".thisisdeletable").toHaveCount(4, { message: "records should be deletable" });
});

test.tags("desktop");
test("kanban grouped by many2one: false column is folded by default", async () => {
    Partner._records[0].product_id = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(3);
    expect(".o_column_folded").toHaveCount(1);
    expect(queryAllTexts(".o_kanban_header")).toEqual(["None\n(1)", "hello\n(1)", "xmo\n(2)"]);

    await contains(".o_kanban_header").click();

    expect(".o_column_folded").toHaveCount(0);
    expect(queryAllTexts(".o_kanban_header")).toEqual(["None\n(1)", "hello\n(1)", "xmo\n(2)"]);

    // reload -> None column should remain open
    await click(".o_searchview_input");
    await press("Enter");
    await animationFrame();

    expect(".o_column_folded").toHaveCount(0);
    expect(queryAllTexts(".o_kanban_header")).toEqual(["None\n(1)", "hello\n(1)", "xmo\n(2)"]);
});

test.tags("desktop");
test("kanban grouped by stage_id: move record from to the None column", async () => {
    // Fake model: partner.stage (only for group headers)
    const Stage = class extends models.Model {
        _name = "partner.stage";
        name = fields.Char();
        _records = [{ id: 10, name: "New" }];
    };
    defineModels([Stage]);

    // Set up a record with no stage initially
    Partner._records = [
        { id: 1, foo: "Task A", stage_id: false },
        { id: 2, foo: "Task B", stage_id: 10 },
    ];
    Partner._fields.stage_id = fields.Many2one({ relation: "partner.stage" });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["stage_id"],
    });

    expect(queryAll(".o_kanban_group")).toHaveCount(2); // None and New

    await click(".o_kanban_group:first .o_kanban_header");

    // Drag a record to the "None" column
    const dragActions = await contains(".o_kanban_record:contains(Task B)").drag();
    await dragActions.moveTo(".o_kanban_group:nth-child(1) .o_kanban_header");
    await dragActions.drop();

    // Reload
    await validateSearch();

    // Assert it's back in "None"
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(queryAllTexts(".o_kanban_record", { root: getKanbanColumn(0) })[1]).toBe("Task B");
});

test("many2many_tags in kanban views", async () => {
    Partner._records[0].category_ids = [6, 7];
    Partner._records[1].category_ids = [7, 8];
    Category._records.push({
        id: 8,
        name: "hello",
        color: 0,
    });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="category_ids" widget="many2many_tags" options="{'color_field': 'color', 'on_tag_click': 'edit_color'}"/>
                        <field name="foo"/>
                        <field name="state" widget="priority"/>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect(resId).toBe(1, {
                message: "should trigger an event to open the clicked record in a form view",
            });
        },
    });

    expect(
        queryAll(".o_field_many2many_tags .o_tag", { root: getKanbanRecord({ index: 0 }) })
    ).toHaveCount(2, {
        message: "first record should contain 2 tags",
    });
    expect(queryAll(".o_tag.o_tag_color_2", { root: getKanbanRecord({ index: 0 }) })).toHaveCount(
        1,
        {
            message: "first tag should have color 2",
        }
    );
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);

    // Checks that second records has only one tag as one should be hidden (color 0)
    expect(".o_kanban_record:nth-child(2) .o_tag").toHaveCount(1, {
        message: "there should be only one tag in second record",
    });
    expect(".o_kanban_record:nth-child(2) .o_tag:first").toHaveText("silver");

    // Write on the record using the priority widget to trigger a re-render in readonly
    await contains(".o_kanban_record:first-child .o_priority_star:first-child").click();

    expect.verifySteps(["web_save"]);
    expect(".o_kanban_record:first-child .o_field_many2many_tags .o_tag").toHaveCount(2, {
        message: "first record should still contain only 2 tags",
    });
    const tags = queryAll(".o_kanban_record:first-child .o_tag");
    expect(tags[0]).toHaveText("gold");
    expect(tags[1]).toHaveText("silver");

    // click on a tag (should trigger switch_view)
    await contains(".o_kanban_record:first-child .o_tag:first-child").click();
});

test("priority field should not be editable when missing access rights", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban edit="0">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="state" widget="priority"/>
                    </t>
                </templates>
            </kanban>`,
    });
    // Try to fill one star in the priority field of the first record
    await contains(".o_kanban_record:first-child .o_priority_star:first-child").click();
    expect(".o_kanban_record:first-child .o_priority .fa-star-o").toHaveCount(2, {
        message: "first record should still contain 2 empty stars",
    });
});

test("Do not open record when clicking on `a` with `href`", async () => {
    expect.assertions(6);

    Partner._records = [{ id: 1, foo: "yop" }];

    mockService("action", {
        async switchView() {
            // when clicking on a record in kanban view,
            // it switches to form view.
            expect.step("switchView");
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <a class="o_test_link" href="#">test link</a>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect(".o_kanban_record a").toHaveCount(1);

    expect(".o_kanban_record a").toHaveAttribute("href", null, {
        message: "link inside kanban record should have non-empty href",
    });

    // Prevent the browser default behaviour when clicking on anything.
    // This includes clicking on a `<a>` with `href`, so that it does not
    // change the URL in the address bar.
    // Note that we should not specify a click listener on 'a', otherwise
    // it may influence the kanban record global click handler to not open
    // the record.
    const testLink = queryFirst(".o_kanban_record a");
    testLink.addEventListener("click", (ev) => {
        expect(ev.defaultPrevented).toBe(false, {
            message: "should not prevented browser default behaviour beforehand",
        });
        expect(ev.target).toBe(testLink, {
            message: "should have clicked on the test link in the kanban record",
        });
        ev.preventDefault();
    });

    await click(".o_kanban_record a");

    expect.verifySteps([]);
});

test("Open record when clicking on widget field", async function (assert) {
    expect.assertions(2);

    Product._views["form"] = `<form string="Product"><field name="display_name"/></form>`;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="salary" widget="monetary"/>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect(resId).toBe(1, { message: "should trigger an event to open the form view" });
        },
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    await click(".o_field_monetary[name=salary]");
});

test("o2m loaded in only one batch", async () => {
    class Subtask extends models.Model {
        _name = "subtask";

        name = fields.Char();

        _records = [
            { id: 1, name: "subtask #1" },
            { id: 2, name: "subtask #2" },
        ];
    }
    defineModels([Subtask]);
    Partner._fields.subtask_ids = fields.One2many({ relation: "subtask" });
    Partner._records[0].subtask_ids = [1];
    Partner._records[1].subtask_ids = [2];

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="subtask_ids" widget="many2many_tags"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    await validateSearch();
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
        "web_read_group",
    ]);
});

test.tags("desktop");
test("kanban with many2many, load and reload", async () => {
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="category_ids" widget="many2many_tags"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    await press("Enter"); // reload
    await animationFrame();

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
        "web_read_group",
    ]);
});

test.tags("desktop");
test("kanban with reference field", async () => {
    Partner._fields.ref_product = fields.Reference({ selection: [["product", "Product"]] });
    Partner._records[0].ref_product = "product,3";
    Partner._records[1].ref_product = "product,5";

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        groupBy: ["product_id"],
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="ref_product"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await press("Enter"); // reload
    await animationFrame();

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
        "web_read_group",
    ]);
    expect(queryAllTexts(".o_kanban_record span")).toEqual(["hello", "", "xmo", ""]);
});

test.tags("desktop");
test("drag and drop a record with load more", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="1">
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });

    expect(queryAllTexts(".o_kanban_group:eq(0) .o_kanban_record")).toEqual(["4"]);
    expect(queryAllTexts(".o_kanban_group:eq(1) .o_kanban_record")).toEqual(["1"]);

    await contains(".o_kanban_group:eq(1) .o_kanban_record").dragAndDrop(".o_kanban_group:eq(0)");
    expect(queryAllTexts(".o_kanban_group:eq(0) .o_kanban_record")).toEqual(["4", "1"]);
    expect(queryAllTexts(".o_kanban_group:eq(1) .o_kanban_record")).toEqual(["2"]);
});

test.tags("desktop");
test("can drag and drop a record from one column to the next", async () => {
    onRpc("web_resequence", () => {
        expect.step("resequence");
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <t t-if="widget.editable">
                            <span class="thisiseditable">edit</span>
                        </t>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(".thisiseditable").toHaveCount(4);

    expect.verifySteps([]);

    // first record of first column moved to the bottom of second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect(".thisiseditable").toHaveCount(4);

    expect.verifySteps(["resequence"]);
});

test.tags("desktop");
test("user without permission cannot drag and drop a column thus sequence remains unchanged on drag and drop attempt", async () => {
    expect.errors(1);

    onRpc("web_resequence", () => {
        throw makeServerError({ message: "No Permission" }); // Simulate user without permission
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });

    expect(queryAllTexts(".o_column_title")).toEqual(["hello\n(2)", "xmo\n(2)"]);

    const groups = queryAll(".o_column_title");
    await contains(groups[0]).dragAndDrop(groups[1]);

    expect(queryAllTexts(".o_column_title")).toEqual(["hello\n(2)", "xmo\n(2)"]);

    expect.verifyErrors(["No Permission"]);
});

test.tags("desktop");
test("user without permission cannot drag and drop a record thus sequence remains unchanged on drag and drop attempt", async () => {
    expect.errors(1);

    onRpc("partner", "web_save", () => {
        throw makeServerError({ message: "No Permission" }); // Simulate user without permission
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_record:first").toHaveText("yop", {
        message: "Checking the initial state of the view",
    });

    await contains(".o_kanban_record").dragAndDrop(".o_kanban_group:nth-child(2)");

    expect(".o_kanban_record:first").toHaveText("yop", {
        message: "Do not let the user d&d the record without permission",
    });

    await contains(".o_kanban_record").dragAndDrop(".o_kanban_record:nth-child(3)");

    expect(".o_kanban_record:first").toHaveText("gnap", {
        message: "Check that the record does not become static after d&d",
    });

    expect.verifyErrors(["No Permission"]);
});

test.tags("desktop");
test("drag and drop highlight on hover and has visible placeholder", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    // first record of first column moved to the bottom of second column
    const { drop, moveTo } = await contains(".o_kanban_group:first-child .o_kanban_record").drag();
    await moveTo(".o_kanban_group:nth-child(2)");
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3); // includes the placeholder
    expect(".o_kanban_group:nth-child(2) .o_kanban_record:eq(2)").toBeVisible();

    expect(getKanbanColumn(1)).toHaveClass("o_kanban_hover");

    await drop();

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect(".o_kanban_group:nth-child(2).o_kanban_hover").toHaveCount(0);
});

test("drag and drop outside of a column", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    // first record of first column moved to the right of a column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_column_quick_create"
    );
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
});

test.tags("desktop");
test("drag and drop a record, grouped by selection", async () => {
    onRpc("web_resequence", () => {
        expect.step("resequence");
    });
    onRpc("partner", "web_save", ({ args }) => {
        expect.step(args[1]);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="state"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["state"],
    });
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect.verifySteps([]);

    // first record of second column moved to the bottom of first column
    await contains(".o_kanban_group:nth-child(2) .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first-child"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(0);
    expect.verifySteps([{ state: "abc" }, "resequence"]);
});

test.tags("desktop");
test("prevent drag and drop of record if grouped by readonly", async () => {
    // Whether the kanban is grouped by state, foo, bar or product_id
    // the user must not be able to drag and drop from one group to another,
    // as state, foo bar, product_id are made readonly one way or another.
    // state must not be draggable:
    // state is not readonly in the model. state is passed in the arch specifying readonly="1".
    // foo must not be draggable:
    // foo is readonly in the model fields. foo is passed in the arch but without specifying readonly.
    // bar must not be draggable:
    // bar is readonly in the model fields. bar is not passed in the arch.
    // product_id must not be draggable:
    // product_id is readonly in the model fields. product_id is passed in the arch specifying readonly="0",
    // but the readonly in the model takes over.
    Partner._fields.foo = fields.Char({ readonly: true });
    Partner._fields.bar = fields.Boolean({ readonly: true });
    Partner._fields.product_id = fields.Many2one({ relation: "product", readonly: true });

    onRpc("partner", "write", () => {
        expect.step("should not be called");
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <div>
                            <field name="foo"/>
                            <field name="product_id" readonly="0" invisible="1"/>
                            <field name="state" readonly="1"/>
                        </div>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_foo" domain="[]" string="GroupBy Foo" context="{ 'group_by': 'foo' }"/>
                <filter name="group_by_bar" domain="[]" string="GroupBy Bar" context="{ 'group_by': 'bar' }"/>
                <filter name="group_by_product" domain="[]" string="GroupBy Product" context="{ 'group_by': 'product_id' }"/>
            </search>`,
        groupBy: ["state"],
    });

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);

    // first record of first column moved to the bottom of second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not be draggable
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Foo");

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(1);

    // first record of first column moved to the bottom of second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not be draggable
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(1);

    expect(getKanbanRecordTexts(0)).toEqual(["blipDEF", "blipGHI"]);

    // second record of first column moved at first place
    await contains(".o_kanban_group:first-child .o_kanban_record:last-of-type").dragAndDrop(
        ".o_kanban_group:first-child .o_kanban_record"
    );

    // should still be able to resequence
    expect(getKanbanRecordTexts(0)).toEqual(["blipGHI", "blipDEF"]);

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Foo");
    await toggleMenuItem("GroupBy Bar");

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(0);

    expect(getKanbanRecordTexts(0)).toEqual(["blipGHI"]);

    // first record of first column moved to the bottom of second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not be draggable
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(0);

    expect(getKanbanRecordTexts(0)).toEqual(["blipGHI"]);

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Bar");
    await toggleMenuItem("GroupBy Product");

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(0);

    expect(getKanbanRecordTexts(0)).toEqual(["yopABC", "gnapGHI"]);

    // first record of first column moved to the bottom of second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not be draggable
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(0);

    expect(getKanbanRecordTexts(0)).toEqual(["yopABC", "gnapGHI"]);
    expect.verifySteps([]);
});

test("prevent drag and drop if grouped by date/datetime field", async () => {
    Partner._records[0].date = "2017-01-08";
    Partner._records[1].date = "2017-01-09";
    Partner._records[2].date = "2017-02-08";
    Partner._records[3].date = "2017-02-10";
    Partner._records[0].datetime = "2017-01-08 10:55:05";
    Partner._records[1].datetime = "2017-01-09 11:31:10";
    Partner._records[2].datetime = "2017-02-08 09:20:25";
    Partner._records[3].datetime = "2017-02-10 08:05:51";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_datetime" domain="[]" string="GroupBy Datetime" context="{ 'group_by': 'datetime' }"/>
            </search>`,
        groupBy: ["date:month"],
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "1st column should contain 2 records of January month",
    });
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "2nd column should contain 2 records of February month",
    });

    // drag&drop a record in another column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not drag&drop record
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2, {
        message: "Should remain same records in first column (2 records)",
    });
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "Should remain same records in 2nd column (2 record)",
    });

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Datetime");
    await toggleMenuItemOption("GroupBy Datetime", "Month");

    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2, {
        message: "1st column should contain 2 records of January month",
    });
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "2nd column should contain 2 records of February month",
    });

    // drag&drop a record in another column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not drag&drop record
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2, {
        message: "Should remain same records in first column(2 records)",
    });
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "Should remain same records in 2nd column(2 record)",
    });
});

test.tags("desktop");
test("prevent drag and drop if grouped by many2many field", async () => {
    Partner._records[0].category_ids = [6, 7];
    Partner._records[3].category_ids = [7];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_state" domain="[]" string="GroupBy State" context="{ 'group_by': 'state' }"/>
            </search>`,
        groupBy: ["category_ids"],
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:first-child .o_column_title:first").toHaveText("gold\n(2)", {
        message: "first column should have correct title",
    });
    expect(".o_kanban_group:last-child .o_column_title:first").toHaveText("silver\n(3)", {
        message: "second column should have correct title",
    });
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:last-child .o_kanban_record").toHaveCount(3);

    // drag&drop a record in another column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:last-child .o_kanban_record").toHaveCount(3);

    // Sanity check: groupby a non m2m field and check dragdrop is working
    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy State");

    expect(".o_kanban_group").toHaveCount(3);
    expect(queryAllTexts(".o_kanban_group .o_column_title")).toEqual([
        "ABC\n(1)",
        "DEF\n(1)",
        "GHI\n(2)",
    ]);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1, {
        message: "first column should have 1 record",
    });
    expect(".o_kanban_group:last-child .o_kanban_record").toHaveCount(2, {
        message: "last column should have 2 records",
    });

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:last-child"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(0, {
        message: "first column should not contain records",
    });
    expect(".o_kanban_group:last-child .o_kanban_record").toHaveCount(3, {
        message: "last column should contain 3 records",
    });
});

test("completely prevent drag and drop if records_draggable set to false", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban records_draggable="false">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    // testing initial state
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(getKanbanRecordTexts()).toEqual(["yop", "gnap", "blip", "blip"]);
    expect(".o_draggable").toHaveCount(0);

    // attempt to drag&drop a record in another column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not drag&drop record
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2, {
        message: "First column should still contain 2 records",
    });
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2, {
        message: "Second column should still contain 2 records",
    });
    expect(getKanbanRecordTexts()).toEqual(["yop", "gnap", "blip", "blip"], {
        message: "Records should not have moved",
    });

    // attempt to drag&drop a record in the same column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first-child .o_kanban_record:last-of-type"
    );

    expect(getKanbanRecordTexts()).toEqual(["yop", "gnap", "blip", "blip"], {
        message: "Records should not have moved",
    });
});

test.tags("desktop");
test("prevent drag and drop of record if save fails", async () => {
    expect.errors(1);

    onRpc("partner", "web_save", () => {
        throw new Error("Save failed");
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="product_id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    // drag&drop a record in another column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    // should not be dropped, card should reset back to first column
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    expect.verifyErrors(["Save failed"]);
});

test("kanban view with default_group_by", async () => {
    expect.assertions(11);

    Partner._records[0].product_id = 1;
    Product._records.push({ id: 1, display_name: "third product" });

    let readGroupCount = 0;
    onRpc("web_read_group", ({ kwargs }) => {
        readGroupCount++;
        switch (readGroupCount) {
            case 1: {
                expect(kwargs.groupby).toEqual(["bar"]);
                break;
            }
            case 2: {
                expect(kwargs.groupby).toEqual(["product_id"]);
                break;
            }
            case 3: {
                expect(kwargs.groupby).toEqual(["bar"]);
                break;
            }
        }
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="bar">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_product_id" domain="[]" string="GroupBy Product" context="{ 'group_by': 'product_id' }"/>
            </search>`,
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_kanban_group").toHaveCount(2);
    // open search bar in mobile
    if (queryAll(".o_control_panel_navigation > button").length) {
        await contains(".o_control_panel_navigation > button").click();
    }
    expect(`.o_searchview_facet`).toHaveCount(0);

    // simulate an update coming from the searchview, with another groupby given
    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Product");
    expect(".o_kanban_group").toHaveCount(3);
    expect(`.o_searchview_facet`).toHaveCount(1);
    expect(`.o_searchview_facet`).toHaveText("GroupBy Product");

    // simulate an update coming from the searchview, removing the previously set groupby
    await contains(".o_searchview_facet .o_facet_remove").click();
    expect(".o_kanban_group").toHaveCount(2);
    expect(`.o_searchview_facet`).toHaveCount(0);
});

test.tags("desktop");
test("edit a favorite: group by = default_group_by", async () => {
    expect.assertions(4);

    const irFilters = [
        {
            context: "{ 'group_by': ['bar'] }",
            domain: "[]",
            id: 1,
            is_default: true,
            name: "My favorite",
            sort: "[]",
            user_ids: [2],
        },
    ];

    onRpc("web_read_group", ({ kwargs }) => {
        expect(kwargs.groupby).toEqual(["bar"]);
    });
    onRpc("/web/domain/validate", () => true);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="bar">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        irFilters,
    });

    expect(getFacetTexts()).toEqual(["My favorite"]);

    await contains(".o_searchview_facet_label").click();
    await addNewRule();
    await contains("button:contains('Search')").click();
    expect(getFacetTexts()).toEqual(["Id = 1"]);
});

test.tags("desktop");
test("edit a favorite: group by != default_group_by", async () => {
    expect.assertions(4);

    const irFilters = [
        {
            context: "{ 'group_by': ['product_id'] }",
            domain: "[]",
            id: 1,
            is_default: true,
            name: "My favorite",
            sort: "[]",
            user_ids: [2],
        },
    ];

    onRpc("web_read_group", ({ kwargs }) => {
        expect(kwargs.groupby).toEqual(["product_id"]);
    });
    onRpc("/web/domain/validate", () => true);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="bar">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        irFilters,
    });

    expect(getFacetTexts()).toEqual(["My favorite"]);

    await contains(".o_searchview_facet_label").click();
    await addNewRule();
    await contains("button:contains('Search')").click();
    expect(getFacetTexts()).toEqual(["Id = 1", "Product"]);
});

test.tags("desktop");
test("kanban view not groupable", async () => {
    patchWithCleanup(kanbanView, { searchMenuTypes: ["filter", "favorite"] });

    onRpc("web_read_group", () => {
        expect.step("web_read_group");
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="bar">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter string="Filter" name="filter" domain="[]"/>
                <filter string="candle" name="itsName" context="{'group_by': 'foo'}"/>
            </search>`,
        context: { search_default_itsName: 1 },
    });

    expect(".o_kanban_renderer").not.toHaveClass("o_kanban_grouped");
    expect(".o_control_panel div.o_search_options div.o_group_by_menu").toHaveCount(0);
    expect(getFacetTexts()).toEqual([]);

    // validate presence of the search arch info
    await toggleSearchBarMenu();
    expect(".o_filter_menu .o_menu_item").toHaveCount(2);
    expect.verifySteps([]);
});

test("kanban view with create=False", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban create="0">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o-kanban-button-new").toHaveCount(0);
});

test("kanban view with create=False and groupby", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban create="0">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o-kanban-button-new").toHaveCount(0);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_quick_add").toHaveCount(0);
});

test("clicking on a link triggers correct event", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <a type="open">Edit</a>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect(resId).toBe(1);
        },
    });
    await contains("a", { root: getKanbanRecord({ index: 0 }) }).click();
});

test.tags("desktop");
test("environment is updated when (un)folding groups", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    // fold the second group and check that the res_ids it contains are no
    // longer in the environment
    const clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Fold");

    expect(getKanbanRecordTexts()).toEqual(["1", "3"]);

    // re-open the second group and check that the res_ids it contains are
    // back in the environment
    await contains(getKanbanColumn(1)).click();

    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);
});

test.tags("desktop");
test("create a column in default grouped on m2o", async () => {
    onRpc("web_resequence", ({ args, method }) => {
        expect.step([method, args[0]]);
    });
    onRpc("name_create", ({ method }) => {
        expect.step(method);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" on_create="quick_create">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_column_quick_create").toHaveCount(1, {
        message: "should have a quick create column",
    });
    expect(".o_column_quick_create input").toHaveCount(0, {
        message: "the input should not be visible",
    });

    await quickCreateKanbanColumn();

    expect(".o_column_quick_create input").toHaveCount(1, {
        message: "the input should be visible",
    });

    // discard the column creation and click it again
    await press("Escape");
    await animationFrame();

    expect(".o_column_quick_create input").toHaveCount(0, {
        message: "the input should not be visible",
    });

    await quickCreateKanbanColumn();

    expect(".o_column_quick_create input").toHaveCount(1, {
        message: "the input should be visible",
    });

    await editKanbanColumnName("new value");
    await validateKanbanColumn();

    expect(".o_kanban_group").toHaveCount(3);
    expect(
        queryAll(".o_column_title:contains(new value)", { root: getKanbanColumn(2) })
    ).toHaveCount(1, {
        message: "the last column should be the newly created one",
    });
    expect(!!getKanbanColumn(2).dataset.id).toBe(true, {
        message: "the created column should have an associated id",
    });
    expect(getKanbanColumn(2)).not.toHaveClass("o_column_folded", {
        message: "the created column should not be folded",
    });
    expect.verifySteps(["name_create", ["web_resequence", [3, 5, 6]]]);

    // fold and unfold the created column, and check that no RPCs are done (as there are no records)
    const clickColumnAction = await toggleKanbanColumnActions(2);
    await clickColumnAction("Fold");

    expect(getKanbanColumn(2)).toHaveClass("o_column_folded");

    await click(getKanbanColumn(2));
    await animationFrame();

    expect(getKanbanColumn(1)).not.toHaveClass("o_column_folded");
    // no rpc should have been done when folding/unfolding
    expect.verifySteps([]);

    // quick create a record
    await createKanbanRecord();

    expect(queryOne(".o_kanban_quick_create", { root: getKanbanColumn(0) })).toHaveCount(1);
});

test("create a column in grouped on m2o without sequence field on view model", async () => {
    delete Partner._fields.sequence;

    onRpc("name_create", () => {
        expect.step("name_create");
    });
    onRpc("web_resequence", ({ args }) => {
        expect.step(["resequence", args[0]]);
        return [];
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_column_quick_create").toHaveCount(1, {
        message: "should have a quick create column",
    });
    expect(".o_column_quick_create input").toHaveCount(0, {
        message: "the input should not be visible",
    });

    await quickCreateKanbanColumn();
    await editKanbanColumnName("new value");
    await validateKanbanColumn();

    expect.verifySteps(["name_create", ["resequence", [3, 5, 6]]]);
});

test.tags("desktop");
test("delete a column in grouped on m2o", async () => {
    stepAllNetworkCalls();
    let resequencedIDs = [];
    onRpc("web_resequence", ({ args }) => {
        resequencedIDs = args[0];
        expect(resequencedIDs.filter(isNaN)).toHaveLength(0, {
            message: "column resequenced should be existing records with IDs",
        });
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" class="o_kanban_test">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    // check the initial rendering
    expect(".o_kanban_group").toHaveCount(2, { message: "should have two columns" });
    expect(queryText(".o_column_title", { root: getKanbanColumn(0) })).toBe("hello\n(2)");
    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("xmo\n(2)");
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2, {
        message: "second column should have two records",
    });

    // check available actions in kanban header's config dropdown
    await toggleKanbanColumnActions(0);
    expect(queryAll(".o_kanban_toggle_fold", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(
        1,
        {
            message: "should be able to fold the column",
        }
    );
    expect(queryAll(".o_group_edit", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(1, {
        message: "should be able to edit the column",
    });
    expect(queryAll(".o_group_delete", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(1, {
        message: "should be able to delete the column",
    });
    expect(
        queryAll(".o_column_archive_records", { root: getKanbanColumnDropdownMenu(0) })
    ).toHaveCount(0, { message: "should not be able to archive all the records" });
    expect(queryAll(".o_column_unarchive_records", { root: getKanbanColumn(0) })).toHaveCount(0, {
        message: "should not be able to restore all the records",
    });

    // delete second column (first cancel the confirm request, then confirm)
    let clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Delete");

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-secondary").click();

    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("xmo\n(2)");

    clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Delete");

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-primary").click();

    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("hello\n(2)");
    expect(".o_kanban_group").toHaveCount(2, { message: "should still have two columns" });
    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("None\n(2)", {
        message: "first column should have no id (Undefined column)",
    });

    // check available actions on 'Undefined' column
    await click(getKanbanColumn(0));
    await animationFrame();
    await toggleKanbanColumnActions(0);

    expect(queryAll(".o_kanban_toggle_fold", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(
        1,
        {
            message: "should be able to fold the column",
        }
    );
    expect(queryAll(".o_group_edit", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(0, {
        message: "should be able to edit the column",
    });
    expect(queryAll(".o_group_delete", { root: getKanbanColumnDropdownMenu(0) })).toHaveCount(0, {
        message: "should not be able to delete the column",
    });
    expect(
        queryAll(".o_column_archive_records", { root: getKanbanColumnDropdownMenu(0) })
    ).toHaveCount(0, { message: "should not be able to archive all the records" });
    expect(
        queryAll(".o_column_unarchive_records", { root: getKanbanColumnDropdownMenu(0) })
    ).toHaveCount(0, { message: "should not be able to restore all the records" });
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
        "unlink",
        "web_read_group",
        "web_search_read",
    ]);
    expect(".o_kanban_group").toHaveCount(2, {
        message: "the old groups should have been correctly deleted",
    });

    // test column drag and drop having an 'Undefined' column
    expect(getKanbanColumn(0)).not.toHaveClass("o_group_draggable");
    await contains(".o_kanban_group:first-child .o_column_title").dragAndDrop(
        queryAll(".o_kanban_group")[1]
    );

    expect(resequencedIDs).toEqual([], {
        message: "resequencing require at least 2 not Undefined columns",
    });

    await quickCreateKanbanColumn();
    await editKanbanColumnName("once third column");
    await validateKanbanColumn();

    expect.verifySteps(["name_create", "web_resequence"]);
    expect(resequencedIDs).toEqual([3, 4], {
        message: "creating a column should trigger a resequence",
    });

    await contains(".o_kanban_group:first-child .o_column_title").dragAndDrop(
        queryAll(".o_kanban_group")[2]
    );

    expect(resequencedIDs).toEqual([3, 4], {
        message: "moving the Undefined column should not affect order of other columns",
    });

    expect(getKanbanColumn(1)).toHaveClass("o_group_draggable");
    await contains(".o_kanban_group:nth-child(2) .o_column_title").dragAndDrop(
        queryAll(".o_kanban_group")[2]
    );
    expect.verifySteps(["web_resequence"]);
    expect(resequencedIDs).toEqual([4, 3], {
        message: "moved column should be resequenced accordingly",
    });
});

test("create a column, delete it and create another one", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_group").toHaveCount(2);

    await quickCreateKanbanColumn();
    await editKanbanColumnName("new column 1");
    await validateKanbanColumn();

    expect(".o_kanban_group").toHaveCount(3);

    const clickColumnAction = await toggleKanbanColumnActions(2);
    await clickColumnAction("Delete");

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-primary").click();

    expect(".o_kanban_group").toHaveCount(2);

    await quickCreateKanbanColumn();
    await editKanbanColumnName("new column 2");
    await validateKanbanColumn();

    expect(".o_kanban_group").toHaveCount(3);
    expect(getKanbanColumn(2).querySelector("div")).toHaveText("new column 2\n(0)", {
        message: "the last column should be the newly created one",
    });
});

test("delete an empty column, then a column with records.", async () => {
    let firstLoad = true;

    onRpc("web_read_group", function ({ parent }) {
        // override web_read_group to return an extra empty groups
        const result = parent();
        if (firstLoad) {
            result.groups.unshift({
                __extra_domain: [["product_id", "=", 7]],
                product_id: [7, "empty group"],
                __count: 0,
                __records: [],
            });
            result.length = 3;
            firstLoad = false;
        }
        return result;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_header .o_column_title:contains('empty group')").toHaveCount(1);
    expect(".o_kanban_header .o_column_title:contains('hello')").toHaveCount(1);
    expect(".o_kanban_header .o_column_title:contains('None')").toHaveCount(0);

    // Delete the empty group
    let clickColumnAction = await toggleKanbanColumnActions();
    await clickColumnAction("Delete");

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-primary").click();

    // Delete the group 'hello'
    clickColumnAction = await toggleKanbanColumnActions();
    await clickColumnAction("Delete");

    expect(".o_dialog").toHaveCount(1);
    await contains(".o_dialog footer .btn-primary").click();

    // None of the previous groups should be present inside the view. Instead, a 'none' column should be displayed.
    expect(".o_kanban_header span:contains('empty group')").toHaveCount(0);
    expect(".o_kanban_header span:contains('hello')").toHaveCount(0);
    expect(".o_kanban_header .o_column_title:contains('None')").toHaveCount(1);
});

test.tags("desktop");
test("edit a column in grouped on m2o", async () => {
    Product._views["form"] = `
        <form string="Product">
            <field name="name"/>
        </form>`;

    onRpc(() => {
        nbRPCs++;
    });

    let nbRPCs = 0;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("xmo\n(2)");

    // edit the title of column [5, 'xmo'] and close without saving
    let clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Edit");

    expect(".modal .o_form_editable").toHaveCount(1);
    expect(".modal .o_form_editable input").toHaveValue("xmo");

    await contains(".modal .o_form_editable input").edit("ged");
    nbRPCs = 0;
    await contains(".modal-header .btn-close").click();

    expect(".modal").toHaveCount(0);
    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("xmo\n(2)");
    expect(nbRPCs).toBe(0, { message: "no RPC should have been done" });

    // edit the title of column [5, 'xmo'] and discard
    clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Edit");
    await contains(".modal .o_form_editable input").edit("ged");
    nbRPCs = 0;
    await contains(".modal button.o_form_button_cancel").click();

    expect(".modal").toHaveCount(0);
    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("xmo\n(2)");
    expect(nbRPCs).toBe(0, { message: "no RPC should have been done" });

    // edit the title of column [5, 'xmo'] and save
    clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Edit");
    await contains(".modal .o_form_editable input").edit("ged");
    nbRPCs = 0;
    await click(".modal .o_form_button_save"); // click on save
    await animationFrame();

    expect(".modal").toHaveCount(0, { message: "the modal should be closed" });
    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("ged\n(2)");
    expect(nbRPCs).toBe(2, {
        message: "should have done 1 write, 1 web_read_group",
    });
});

test("edit a column propagates right context", async () => {
    expect.assertions(3);

    Product._views["form"] = `
        <form string="Product">
            <field name="display_name"/>
        </form>`;

    serverState.lang = "nb_NO";

    onRpc(({ method, model, kwargs }) => {
        if (model === "partner" && method === "web_read_group") {
            expect(kwargs.context.lang).toBe("nb_NO", {
                message: "lang is present in context for partner operations",
            });
        } else if (model === "product") {
            expect(kwargs.context.lang).toBe("nb_NO", {
                message: "lang is present in context for product operations",
            });
        }
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    const clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Edit");
});

test("quick create column should be opened if there is no column", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        domain: [["foo", "=", "norecord"]],
    });

    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_column_quick_create").toHaveCount(1);
    expect(".o_column_quick_create input").toHaveCount(1, {
        message: "the quick create should be opened",
    });
});

test("quick create column should close on window click if there is no column", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        domain: [["foo", "=", "norecord"]],
    });

    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_column_quick_create").toHaveCount(1);
    expect(".o_column_quick_create input").toHaveCount(1, {
        message: "the quick create should be opened",
    });
    // click outside should discard quick create column
    await contains(getFixture()).click();
    expect(".o_column_quick_create input").toHaveCount(0, {
        message: "the quick create should be closed",
    });
});

test("quick create several columns in a row", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_group").toHaveCount(2, { message: "should have two columns" });
    expect(".o_column_quick_create").toHaveCount(1, {
        message: "should have a ColumnQuickCreate widget",
    });
    expect(".o_column_quick_create.o_quick_create_folded:visible").toHaveCount(1, {
        message: "the ColumnQuickCreate should be folded",
    });
    expect(".o_column_quick_create.o_quick_create_unfolded:visible").toHaveCount(0, {
        message: "the ColumnQuickCreate should be folded",
    });

    // add a new column
    await quickCreateKanbanColumn();
    expect(".o_column_quick_create.o_quick_create_folded:visible").toHaveCount(0, {
        message: "the ColumnQuickCreate should be unfolded",
    });
    expect(".o_column_quick_create.o_quick_create_unfolded:visible").toHaveCount(1, {
        message: "the ColumnQuickCreate should be unfolded",
    });
    await editKanbanColumnName("New Column 1");
    await validateKanbanColumn();
    expect(".o_kanban_group").toHaveCount(3, { message: "should now have three columns" });

    // add another column
    expect(".o_column_quick_create.o_quick_create_folded:visible").toHaveCount(0, {
        message: "the ColumnQuickCreate should still be unfolded",
    });
    expect(".o_column_quick_create.o_quick_create_unfolded:visible").toHaveCount(1, {
        message: "the ColumnQuickCreate should still be unfolded",
    });
    await editKanbanColumnName("New Column 2");
    await validateKanbanColumn();
    expect(".o_kanban_group").toHaveCount(4);
});

test.tags("desktop");
test("quick create column with enter", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await quickCreateKanbanColumn();
    await edit("New Column 1");
    await animationFrame();
    expect(".o_kanban_group").toHaveCount(2);

    await press("Enter");
    await animationFrame();
    expect(".o_kanban_group").toHaveCount(3);
});

test.tags("desktop");
test("empty stages kanban examples", async () => {
    Partner._records = [];
    registry.category("kanban_examples").add("test", {
        allowedGroupBys: ["product_id"],
        examples: [
            {
                name: "A first example",
                columns: ["Column 1", "Column 2", "Column 3"],
                description: "A weak description.",
            },
            {
                name: "A second example",
                columns: ["Col 1", "Col 2"],
                description: `A fantastic description.`,
            },
        ],
    });
    after(() => registry.category("kanban_examples").remove("test"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" examples="test">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_group_nocontent").toHaveCount(1, {
        message: "should show the empty stages kanban helper",
    });

    expect(".o_kanban_group_nocontent").toHaveText(
        "No product yet, let's create some!\n\nLack of inspiration? See examples"
    );

    expect(".o_kanban_group_nocontent .o_kanban_examples").toHaveCount(1, {
        message: "should have a link to see examples",
    });

    // click to see the examples
    await contains(".o_kanban_group_nocontent .o_kanban_examples").click();

    expect(".modal .o_kanban_examples_dialog").toHaveCount(1, {
        message: "should have open the examples dialog",
    });
    expect(".modal .o_notebook_headers li").toHaveCount(2, {
        message: "should have two examples (in the menu)",
    });
    expect(".modal .o_notebook_headers").toHaveText("A first example\nA second example", {
        message: "example names should be correct",
    });
    expect(".modal .o_notebook_content .tab-pane").toHaveCount(1, {
        message: "should have only rendered one page",
    });

    const firstPane = queryFirst(".modal .o_notebook_content .tab-pane");
    expect(queryAll(".o_kanban_examples_group", { root: firstPane })).toHaveCount(3);
    expect(queryAllTexts("h6", { root: firstPane })).toEqual(["Column 1", "Column 2", "Column 3"], {
        message: "column titles should be correct",
    });
    expect(queryFirst(".o_kanban_examples_description", { root: firstPane })).toHaveInnerHTML(
        "A weak description.",
        { message: "An escaped description should be displayed" }
    );

    await contains(".nav-item:nth-child(2) .nav-link").click();
    const secondPane = queryFirst(".o_notebook_content");
    expect(queryAll(".o_kanban_examples_group", { root: firstPane })).toHaveCount(2);
    expect(queryAllTexts("h6", { root: secondPane })).toEqual(["Col 1", "Col 2"], {
        message: "column titles should be correct",
    });
    expect(secondPane.querySelector(".o_kanban_examples_description").innerHTML).toBe(
        "A fantastic description.",
        { message: "A formatted description should be displayed." }
    );
});

test("quick create column with x_name as _rec_name", async () => {
    Product._rec_name = "x_name";
    Product._fields.x_name = fields.Char();
    Product._records = [
        { id: 3, x_name: "hello" },
        { id: 5, x_name: "xmo" },
    ];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    await quickCreateKanbanColumn();
    await editKanbanColumnName("New Column 1");
    await validateKanbanColumn();
    expect(".o_kanban_group").toHaveCount(3, { message: "should now have three columns" });
});

test.tags("desktop");
test("count of folded groups in empty kanban with sample data", async () => {
    onRpc("web_read_group", () => ({
        groups: [
            {
                product_id: [1, "New"],
                __count: 0,
                __extra_domain: [],
                __records: [],
            },
            {
                product_id: [2, "In Progress"],
                __count: 0,
                __extra_domain: [],
            },
        ],
        length: 2,
    }));

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
        domain: [["id", "<", 0]],
    });

    expect(queryFirst(".o_content")).toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(2);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(queryAllTexts(".o_kanban_group")).toEqual(["New", "In Progress"]);
});

test.tags("desktop");
test("empty stages kanban examples: with folded columns", async () => {
    registry.category("kanban_examples").add("test", {
        allowedGroupBys: ["product_id"],
        foldField: "folded",
        examples: [
            {
                name: "A first example",
                columns: ["not folded"],
                foldedColumns: ["folded"],
                description: "A weak description.",
            },
        ],
    });
    after(() => registry.category("kanban_examples").remove("test"));

    Partner._records = [];
    Product._fields.folded = fields.Boolean();

    onRpc(["name_create", "write"], ({ model, method, args }) => {
        expect.step(`${method} (model: ${model}):${JSON.stringify(args)}`);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" examples="test">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    // click to see the examples
    await contains(".o_kanban_group_nocontent .o_kanban_examples").click();

    // apply the examples
    expect.verifySteps([]);
    await contains(".modal .modal-footer .btn.btn-primary").click();
    expect.verifySteps([
        'name_create (model: product):["not folded"]',
        'name_create (model: product):["folded"]',
        'write (model: product):[[7],{"folded":true}]',
    ]);

    // the applied examples should be visible
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:not(.o_column_folded)").toHaveCount(1);
    expect(".o_kanban_group.o_column_folded").toHaveCount(1);
    expect(queryAllTexts(".o_kanban_group")).toEqual(["not folded\n(0)", "folded"]);
});

test.tags("desktop");
test("empty stages kanban examples: apply button's display text", async () => {
    Partner._records = [];
    const applyExamplesText = "Use This For My Test";
    registry.category("kanban_examples").add("test", {
        allowedGroupBys: ["product_id"],
        applyExamplesText: applyExamplesText,
        examples: [
            {
                name: "A first example",
                columns: ["Column 1", "Column 2", "Column 3"],
            },
            {
                name: "A second example",
                columns: ["Col 1", "Col 2"],
            },
        ],
    });
    after(() => registry.category("kanban_examples").remove("test"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" examples="test">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    // click to see the examples
    await contains(".o_kanban_group_nocontent .o_kanban_examples").click();

    expect(".modal footer.modal-footer button.btn-primary").toHaveText(applyExamplesText, {
        message: "the primary button should display the value of applyExamplesText",
    });
});

test("nocontent helper after adding a record (kanban with progressbar)", async () => {
    onRpc("web_read_group", () => ({
        groups: [
            {
                __extra_domain: [["product_id", "=", 3]],
                __count: 0,
                product_id: [3, "hello"],
                __records: [],
            },
        ],
    }));
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban >
                <progressbar field="foo" colors='{"yop": "success", "gnap": "warning", "blip": "danger"}' sum_field="int_field"/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
        domain: [["foo", "=", "abcd"]],
        noContentHelp: "No content helper",
    });

    expect(".o_view_nocontent").toHaveCount(1, { message: "the nocontent helper is displayed" });

    // add a record
    await quickCreateKanbanRecord();
    await editKanbanRecordQuickCreateInput("display_name", "twilight sparkle");
    await validateKanbanRecord();

    expect(".o_view_nocontent").toHaveCount(0, {
        message: "the nocontent helper is not displayed after quick create",
    });

    // cancel quick create
    await discardKanbanRecord();
    expect(".o_view_nocontent").toHaveCount(0, {
        message: "the nocontent helper is not displayed after cancelling the quick create",
    });
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "read_progress_bar",
        "web_read_group",
        "has_group",
        "onchange",
        "name_create",
        "onchange",
        "web_read",
        "read_progress_bar",
        "formatted_read_group",
    ]);
});

test.tags("desktop");
test("ungrouped kanban view can be grouped, then ungrouped", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_product" domain="[]" string="GroupBy Product" context="{ 'group_by': 'product_id' }"/>
            </search>`,
    });

    expect(".o_kanban_renderer").not.toHaveClass("o_kanban_grouped");

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Product");

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");

    await toggleMenuItem("GroupBy Product");

    expect(".o_kanban_renderer").not.toHaveClass("o_kanban_grouped");
});

test.tags("desktop");
test("no content helper when no data", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: '<p class="hello">click to add a partner</p>',
    });

    expect(".o_view_nocontent").toHaveCount(1, { message: "should display the no content helper" });

    expect(".o_view_nocontent").toHaveText('<p class="hello">click to add a partner</p>', {
        message: "should have rendered no content helper from action",
    });

    MockServer.env["partner"].create([{ foo: "new record" }]);
    await press("Enter");
    await animationFrame();

    expect(".o_view_nocontent").toHaveCount(0, {
        message: "should not display the no content helper",
    });
});

test("no nocontent helper for grouped kanban with empty groups", async () => {
    onRpc("web_read_group", function ({ kwargs, parent }) {
        // override web_read_group to return empty groups, as this is
        // the case for several models (e.g. project.task grouped
        // by stage_id)
        const result = parent();
        for (const group of result.groups) {
            group.__count = 0;
            group.__records = [];
        }
        return result;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_group").toHaveCount(2, { message: "there should be two columns" });
    expect(".o_kanban_record").toHaveCount(0, { message: "there should be no records" });
});

test("stages nocontent helper for grouped kanban with no records", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_group").toHaveCount(0, { message: "there should be no columns" });
    expect(".o_kanban_record").toHaveCount(0, { message: "there should be no records" });
    expect(".o_view_nocontent.o_kanban_group_nocontent").toHaveCount(1);
    expect(".o_column_quick_create").toHaveCount(1);
});

test("basic nocontent helper is shown when no longer creating column", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: "No content helper",
    });

    expect(".o_view_nocontent.o_kanban_group_nocontent").toHaveCount(1, {
        message: "there should be no nocontent helper (we are in 'column creation mode')",
    });

    // creating a new column
    await editKanbanColumnName("applejack");
    await validateKanbanColumn();

    expect(".o_view_nocontent").toHaveCount(0, {
        message: "there should be no nocontent helper (still in 'column creation mode')",
    });

    // leaving column creation mode
    await press("Escape");
    await animationFrame();

    expect(".o_view_nocontent:not(.o_kanban_group_nocontent)").toHaveCount(1);
});

test("no nocontent helper is hidden when quick creating a column", async () => {
    Partner._records = [];

    onRpc("web_read_group", () => ({
        groups: [
            {
                __extra_domain: [["product_id", "=", 3]],
                __count: 0,
                product_id: [3, "hello"],
            },
        ],
        length: 1,
    }));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: "No content helper",
    });

    expect(".o_view_nocontent").toHaveCount(1, { message: "there should be a nocontent helper" });

    await quickCreateKanbanColumn();

    expect(".o_view_nocontent").toHaveCount(0, {
        message: "there should be no nocontent helper (we are in 'column creation mode')",
    });
});

test("nocontent helper for grouped kanban (on m2o field) with no records with no group_create", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id" group_create="false">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_group").toHaveCount(0, { message: "there should be no columns" });
    expect(".o_kanban_record").toHaveCount(0, { message: "there should be no records" });
    expect(".o_view_nocontent").toHaveCount(0, {
        message: "there should not be a nocontent helper",
    });
    expect(".o_column_quick_create").toHaveCount(0, {
        message: "there should not be a column quick create",
    });
});

test("nocontent helper for grouped kanban (on date field) with no records with no group_create", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="date" group_create="false">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_kanban_record").toHaveCount(0);
    expect(".o_view_nocontent").toHaveCount(1);
    expect(".o_column_quick_create").toHaveCount(0);
    expect(".o_kanban_example_background").toHaveCount(0);
});

test("empty grouped kanban with sample data and no columns", async () => {
    Partner._records = [];

    await mountView({
        arch: `
            <kanban default_group_by="product_id" sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_group_nocontent").toHaveCount(1);
    expect(".o_quick_create_unfolded").toHaveCount(1);
});

test("empty kanban with sample data grouped by date range (fill temporal)", async () => {
    Partner._records = [];

    onRpc("web_read_group", () =>
        // Simulate fill temporal
        ({
            groups: [
                {
                    __count: 0,
                    state: false,
                    "date:month": ["2022-12-01", "December 2022"],
                    __extra_domain: [
                        ["date", ">=", "2022-12-01"],
                        ["date", "<", "2023-01-01"],
                    ],
                    __records: [],
                },
            ],
            length: 1,
        })
    );
    await mountView({
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <field name="int_field"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["date:month"],
        resModel: "partner",
        type: "kanban",
        noContentHelp: "No content helper",
    });

    expect(".o_view_nocontent").toHaveCount(1);
    expect(".o_kanban_group .o_column_title").toHaveText("December 2022");
    expect(".o_kanban_group").toHaveCount(1);
    expect(".o_kanban_group .o_kanban_record").toHaveCount(16);
});

test.tags("desktop");
test("empty grouped kanban with sample data: keynav", async () => {
    onRpc("web_read_group", function ({ parent }) {
        const result = parent();
        result.groups.forEach((g) => (g.__count = 0));
        return result;
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <field name="state" widget="priority"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_record").toHaveCount(16);
    expect(document.activeElement).toHaveClass("o_searchview_input");

    await press("ArrowDown");
    await animationFrame();

    expect(document.activeElement).toHaveClass("o_searchview_input");
});

test.tags("desktop");
test("empty kanban with sample data", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban sample="1">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="no_match" string="Match nothing" domain="[['id', '=', 0]]"/>
            </search>`,
        noContentHelp: "No content helper",
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(10, {
        message: "there should be 10 sample records",
    });
    expect(".o_view_nocontent").toHaveCount(1);

    await toggleSearchBarMenu();
    await toggleMenuItem("Match nothing");

    expect(".o_content").not.toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(0);
    expect(".o_view_nocontent").toHaveCount(1);
});

test("empty grouped kanban with sample data and many2many_tags", async () => {
    onRpc("web_read_group", function ({ kwargs, parent }) {
        const result = parent();
        // override web_read_group to return empty groups, as this is
        // the case for several models (e.g. project.task grouped
        // by stage_id)
        result.groups.forEach((group) => {
            group.__count = 0;
        });
        return result;
    });
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban sample="1">
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                        <field name="category_ids" widget="many2many_tags"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2, { message: "there should be 2 'real' columns" });
    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(queryAll(".o_kanban_record").length >= 1).toBe(true, {
        message: "there should be sample records",
    });
    expect(queryAll(".o_field_many2many_tags .o_tag").length >= 1).toBe(true, {
        message: "there should be tags",
    });
    // should not read the tags
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
    ]);
});

test.tags("desktop");
test("sample data does not change after reload with sample data", async () => {
    Partner._views["kanban"] = `
        <kanban sample="1">
            <templates>
                <t t-name="card">
                    <field name="int_field"/>
                </t>
            </templates>
        </kanban>`;
    // list-view so that there is a view switcher, unused
    Partner._views["list"] = '<list><field name="foo"/></list>';

    onRpc("web_read_group", function ({ kwargs, parent }) {
        const result = parent();
        // override web_read_group to return empty groups, as this is
        // the case for several models (e.g. project.task grouped
        // by stage_id)
        result.groups.forEach((group) => {
            group.__count = 0;
        });
        return result;
    });
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "list"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_group").toHaveCount();
    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_record").toHaveCount(16);

    const kanbanText = queryText(".o_kanban_view");
    await contains(".o_control_panel .o_switch_view.o_kanban").click();

    expect(".o_kanban_view").toHaveText(kanbanText, {
        message: "the content should be the same after reloading the view",
    });
});

test.tags("desktop");
test("non empty kanban with sample data", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban sample="1">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="no_match" string="Match nothing" domain="[['id', '=', 0]]"/>
            </search>`,
        noContentHelp: "No content helper",
    });

    expect(".o_content").not.toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    expect(".o_view_nocontent").toHaveCount(0);

    await toggleSearchBarMenu();
    await toggleMenuItem("Match nothing");

    expect(".o_content").not.toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(0);
});

test("empty grouped kanban with sample data: add a column", async () => {
    onRpc("web_read_group", function ({ parent }) {
        const result = parent();
        result.groups = this.env["product"].map((r) => ({
            product_id: [r.id, r.display_name],
            __count: 0,
            __records: [], // Open group by default
            __extra_domain: [["product_id", "=", r.id]],
        }));
        result.length = result.groups.length;
        return result;
    });

    await mountView({
        arch: `
            <kanban default_group_by="product_id" sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(2);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });

    await quickCreateKanbanColumn();
    await editKanbanColumnName("Yoohoo");
    await validateKanbanColumn();

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(3);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });
});

test.tags("desktop");
test("empty grouped kanban with sample data: cannot fold a column", async () => {
    // folding a column in grouped kanban with sample data is disabled, for the sake of simplicity
    onRpc("web_read_group", function ({ kwargs, parent }) {
        const result = parent();
        // override web_read_group to return a single, empty group
        result.groups = result.groups.slice(0, 1);
        result.groups[0]["__count"] = 0;
        result.length = 1;
        return result;
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(1);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });

    await toggleKanbanColumnActions(0);

    expect(getDropdownMenu(".o_group_config").querySelector(".o_kanban_toggle_fold")).toHaveClass(
        "disabled"
    );
});

test("empty grouped kanban with sample data: delete a column", async () => {
    Partner._records = [];

    let groups = [
        {
            product_id: [1, "New"],
            __count: 0,
            __extra_domain: [],
            __records: [],
        },
    ];

    onRpc("web_read_group", () =>
        // override read_group to return a single, empty group
        ({
            groups,
            length: groups.length,
        })
    );

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban default_group_by="product_id" sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(1);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });

    // Delete the first column
    groups = [];
    const clickColumnAction = await toggleKanbanColumnActions(0);
    await clickColumnAction("Delete");
    await contains(".o_dialog footer .btn-primary").click();

    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_column_quick_create.o_quick_create_unfolded").toHaveCount(1);
});

test("empty grouped kanban with sample data: add a column and delete it right away", async () => {
    onRpc("web_read_group", function ({ parent }) {
        const result = parent();
        result.groups = this.env["product"].map((r) => ({
            product_id: [r.id, r.display_name],
            __count: 0,
            __records: [], // Open group by default
            __extra_domain: [["product_id", "=", r.id]],
        }));
        result.length = result.groups.length;
        return result;
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban default_group_by="product_id" sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(2);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });

    // add a new column
    await quickCreateKanbanColumn();
    await editKanbanColumnName("Yoohoo");
    await validateKanbanColumn();

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(3);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });

    // delete the column we just created
    const clickColumnAction = await toggleKanbanColumnActions(2);
    await clickColumnAction("Delete");
    await contains(".o_dialog footer .btn-primary").click();

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_group").toHaveCount(2);
    expect(queryAll(".o_kanban_record").length > 0).toBe(true, {
        message: "should contain sample records",
    });
});

test.tags("desktop");
test("kanban with sample data: do an on_create action", async () => {
    Partner._records = [];
    Partner._views["form,some_view_ref"] = `<form><field name="foo"/></form>`;

    onRpc("/web/action/load", () => ({
        type: "ir.actions.act_window",
        name: "Archive Action",
        res_model: "partner",
        view_mode: "form",
        target: "new",
        views: [[false, "form"]],
    }));

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1" on_create="myCreateAction">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(10, {
        message: "there should be 10 sample records",
    });
    expect(".o_view_nocontent").toHaveCount(1);

    await createKanbanRecord();
    expect(".modal").toHaveCount(1);

    await contains(".modal .o_form_button_save").click();
    expect(".o_content").not.toHaveClass("o_view_sample_data");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect(".o_view_nocontent").toHaveCount(0);
});

test("kanban with sample data grouped by m2o and existing groups", async () => {
    Partner._records = [];

    onRpc("web_read_group", () => ({
        groups: [
            {
                __count: 0,
                __records: [],
                product_id: [3, "hello"],
                __extra_domain: [["product_id", "=", "3"]],
            },
        ],
        length: 1,
    }));

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="product_id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_view_nocontent").toHaveCount(1);
    expect(".o_kanban_group:first .o_column_title").toHaveText("hello");
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(16);
    expect(".o_kanban_record").toHaveText("hello");
});

test(`kanban grouped by m2o with sample data with more than 5 real groups`, async () => {
    Partner._records = [];
    onRpc("web_read_group", () => ({
        // simulate 6, empty, real groups
        groups: [1, 2, 3, 4, 5, 6].map((id) => ({
            __count: 0,
            __records: [],
            product_id: [id, `Value ${id}`],
            __extra_domain: [["product_id", "=", id]],
        })),
        length: 6,
    }));

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="product_id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(queryAllTexts(`.o_kanban_group .o_column_title`)).toEqual([
        "Value 1",
        "Value 2",
        "Value 3",
        "Value 4",
        "Value 5",
        "Value 6",
    ]);
});

test.tags("desktop");
test("bounce create button when no data and click on empty area", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="no_match" string="Match nothing" domain="[['id', '=', 0]]"/>
            </search>`,
        noContentHelp: "click to add a partner",
    });

    await contains(".o_kanban_view").click();
    expect(".o-kanban-button-new").not.toHaveClass("o_catch_attention");

    await toggleSearchBarMenu();
    await toggleMenuItem("Match nothing");

    await contains(".o_kanban_renderer").click();
    expect(".o-kanban-button-new").toHaveClass("o_catch_attention");
});

test("buttons with modifiers", async () => {
    Partner._records[1].bar = false; // so that test is more complete

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="foo"/>
                <field name="bar"/>
                <field name="state"/>
                <templates>
                    <div t-name="card">
                        <button class="o_btn_test_1" type="object" name="a1" invisible="foo != 'yop'"/>
                        <button class="o_btn_test_2" type="object" name="a2" invisible="bar and state not in ['abc', 'def']"/>
                    </div>
                </templates>
            </kanban>`,
    });

    expect(".o_btn_test_1").toHaveCount(1, { message: "kanban should have one buttons of type 1" });
    expect(".o_btn_test_2").toHaveCount(3, {
        message: "kanban should have three buttons of type 2",
    });
});

test("support styling of anchor tags with action type", async function (assert) {
    expect.assertions(3);

    mockService("action", {
        doActionButton(action) {
            expect(action.name).toBe("42");
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <a type="action" name="42" class="btn-primary" style="margin-left: 10px"><i class="oi oi-arrow-right"/> Click me !</a>
                    </div>
                </templates>
            </kanban>`,
    });

    await click("a[type='action']");
    expect("a[type='action']:first").toHaveClass("btn-primary");
    expect(queryFirst("a[type='action']").style.marginLeft).toBe("10px");
});

test("button executes action and reloads", async () => {
    stepAllNetworkCalls();

    let count = 0;
    mockService("action", {
        async doActionButton({ onClose }) {
            count++;
            await animationFrame();
            onClose();
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <button type="object" name="a1" class="a1">
                            A1
                        </button>
                    </div>
                </templates>
            </kanban>`,
    });

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);
    expect("button.a1").toHaveCount(4);
    expect("button.a1:first").not.toHaveAttribute("disabled");

    await click("button.a1");

    expect("button.a1:first").toHaveAttribute("disabled");

    await animationFrame();

    expect("button.a1:first").not.toHaveAttribute("disabled");
    expect(count).toBe(1, { message: "should have triggered an execute action only once" });
    // the records should be reloaded after executing a button action
    expect.verifySteps(["web_search_read"]);
});

test("button executes action and check domain", async () => {
    Partner._fields.active = fields.Boolean({ default: true });
    for (let i = 0; i < Partner._records.length; i++) {
        Partner._records[i].active = true;
    }

    mockService("action", {
        doActionButton({ onClose }) {
            MockServer.env["partner"][0].active = false;
            onClose();
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <button type="object" name="a1" />
                        <button type="object" name="action_archive" class="action-archive" />
                    </div>
                </templates>
            </kanban>`,
    });

    expect(queryText("span", { root: getKanbanRecord({ index: 0 }) })).toBe("yop", {
        message: "should display 'yop' record",
    });
    await contains("button.action-archive", { root: getKanbanRecord({ index: 0 }) }).click();
    expect(queryText("span", { root: getKanbanRecord({ index: 0 }) })).not.toBe("yop", {
        message: "should have removed 'yop' record from the view",
    });
});

test("field tag with modifiers but no widget", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" invisible="id == 1"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:first").toHaveText("");
    expect(".o_kanban_record:eq(1)").toHaveText("blip");
});

test("field tag with widget and class attributes", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" widget="char" class="hi"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_field_widget.hi").toHaveCount(4);
});

test("rendering date and datetime (value)", async () => {
    Partner._records[0].date = "2017-01-25";
    Partner._records[1].datetime = "2016-12-12 10:55:05";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field class="date" name="date"/>
                        <field class="datetime" name="datetime"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecord({ index: 0 }).querySelector(".date")).toHaveText("Jan 25, 2017");
    expect(getKanbanRecord({ index: 1 }).querySelector(".datetime")).toHaveText(
        "Dec 12, 2016, 11:55 AM"
    );
});

test("rendering date and datetime (raw value)", async () => {
    Partner._records[0].date = "2017-01-25";
    Partner._records[1].datetime = "2016-12-12 10:55:05";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="date"/>
                <field name="datetime"/>
                <templates>
                    <t t-name="card">
                        <span class="date" t-esc="record.date.raw_value"/>
                        <span class="datetime" t-esc="record.datetime.raw_value"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecord({ index: 0 }).querySelector(".date")).toHaveText(
        "2017-01-25T00:00:00.000+01:00"
    );
    expect(getKanbanRecord({ index: 1 }).querySelector(".datetime")).toHaveText(
        "2016-12-12T11:55:05.000+01:00"
    );
});

test("rendering many2one (value)", async () => {
    Partner._records[1].product_id = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="product_id" class="product_id"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual(["hello", "", "hello", "xmo"]);
});

test("rendering many2one (raw value)", async () => {
    Partner._records[1].product_id = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="product_id"/>
                <templates>
                    <t t-name="card">
                        <span class="product_id" t-esc="record.product_id.raw_value"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual(["3", "false", "3", "5"]);
});

test("evaluate conditions on relational fields", async () => {
    Partner._records[0].product_id = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="product_id"/>
                <field name="category_ids"/>
                <templates>
                    <t t-name="card">
                        <button t-if="!record.product_id.raw_value" class="btn_a">A</button>
                        <button t-if="!record.category_ids.raw_value.length" class="btn_b">B</button>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4, {
        message: "there should be 4 records",
    });
    expect(".o_kanban_record:not(.o_kanban_ghost) .btn_a").toHaveCount(1, {
        message: "only 1 of them should have the 'Action' button",
    });
    expect(".o_kanban_record:not(.o_kanban_ghost) .btn_b").toHaveCount(2, {
        message: "only 2 of them should have the 'Action' button",
    });
});

test.tags("desktop");
test("resequence columns in grouped by m2o", async () => {
    Product._fields.sequence = fields.Integer();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("hello\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    await contains(".o_kanban_group:first-child").dragAndDrop(".o_kanban_group:nth-child(2)");

    // Drag & drop on column (not title) should not work
    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("hello\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    await contains(".o_kanban_group:first-child .o_column_title").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("xmo\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["2", "4", "1", "3"]);
});

test.tags("desktop");
test("resequence all when creating new record + partial resequencing", async () => {
    onRpc("web_resequence", ({ args, kwargs }) => {
        const [ids] = args;
        const { field_name: fieldName, offset } = kwargs;
        expect.step({ ids, ...(offset ? { offset } : {}) });
        const resequenceOffset = offset || 0;

        return ids.map((id, index) => ({
            id,
            [fieldName]: resequenceOffset + index,
        }));
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await quickCreateKanbanColumn();
    await editKanbanColumnName("foo");
    await validateKanbanColumn();
    expect.verifySteps([{ ids: [3, 5, 6] }]);

    await editKanbanColumnName("bar");
    await validateKanbanColumn();
    expect.verifySteps([{ ids: [3, 5, 6, 7] }]);

    await editKanbanColumnName("baz");
    await validateKanbanColumn();
    expect.verifySteps([{ ids: [3, 5, 6, 7, 8] }]);

    await editKanbanColumnName("boo");
    await validateKanbanColumn();
    expect.verifySteps([{ ids: [3, 5, 6, 7, 8, 9] }]);

    // When rearranging, only resequence the affected records. In this example,
    // dragging column 2 to column 4 should only resequence [5, 6, 7] to [6, 7, 5]
    // with offset 1.
    await contains(".o_kanban_group:nth-child(2) .o_column_title").dragAndDrop(
        ".o_kanban_group:nth-child(4)"
    );
    expect.verifySteps([{ ids: [6, 7, 5], offset: 1 }]);
});

test("prevent resequence columns if groups_draggable=false", async () => {
    Product._fields.sequence = fields.Integer();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban groups_draggable='0'>
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("hello\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    await contains(".o_kanban_group:first-child").dragAndDrop(".o_kanban_group:nth-child(2)");

    // Drag & drop on column (not title) should not work
    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("hello\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    await contains(".o_kanban_group:first-child .o_column_title").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(getKanbanColumn(0).querySelector(".o_column_title")).toHaveText("hello\n(2)");
    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);
});

test("open config dropdown on kanban with records and groups draggable off", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban groups_draggable='0' records_draggable='0'>
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group .o_group_config").toHaveCount(2);
    expect(".o-dropdown--menu").toHaveCount(0);

    await toggleKanbanColumnActions(0);

    expect(".o-dropdown--menu").toHaveCount(1);
});

test("properly evaluate more complex domains", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="bar"/>
                <field name="category_ids"/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <button type="object" invisible="bar or category_ids" class="btn btn-primary float-end" name="arbitrary">Join</button>
                    </t>
                </templates>
            </kanban>`,
    });

    expect("button.float-end.oe_kanban_action").toHaveCount(1, {
        message: "only one button should be visible",
    });
});

test("kanban with color attribute", async () => {
    Category._records[0].color = 5;
    Category._records[1].color = 6;

    await mountView({
        type: "kanban",
        resModel: "category",
        arch: `
            <kanban highlight_color="color">
                <templates>
                    <t t-name="card">
                        <field name="name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecord({ index: 0 })).toHaveClass("o_kanban_color_5");
    expect(getKanbanRecord({ index: 1 })).toHaveClass("o_kanban_color_6");
});

test("edit the kanban color with the colorpicker", async () => {
    Category._records[0].color = 12;

    onRpc("web_save", ({ args }) => {
        expect.step(`write-color-${args[1].color}`);
    });

    await mountView({
        type: "kanban",
        resModel: "category",
        arch: `
            <kanban highlight_color="color">
                <templates>
                    <t t-name="menu">
                        <field name="color" widget="kanban_color_picker"/>
                    </t>
                    <t t-name="card">
                        <field name="name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await toggleKanbanRecordDropdown(0);

    expect(".o_kanban_record.o_kanban_color_12").toHaveCount(0, {
        message: "no record should have the color 12",
    });
    expect(
        queryAll(".o_kanban_colorpicker", { root: getDropdownMenu(getKanbanRecord({ index: 0 })) })
    ).toHaveCount(1);
    expect(
        queryAll(".o_kanban_colorpicker > *", {
            root: getDropdownMenu(getKanbanRecord({ index: 0 })),
        })
    ).toHaveCount(12, { message: "the color picker should have 12 children (the colors)" });

    await contains(".o_kanban_colorpicker .o_colorlist_item_color_9").click();

    // should write on the color field
    expect.verifySteps(["write-color-9"]);
    expect(getKanbanRecord({ index: 0 })).toHaveClass("o_kanban_color_9");
});

test("kanban with colorpicker and node with color attribute", async () => {
    Category._fields.colorpickerField = fields.Integer();
    Category._records[0].colorpickerField = 3;

    onRpc("web_save", ({ args }) => {
        expect.step(`write-color-${args[1].colorpickerField}`);
    });

    await mountView({
        type: "kanban",
        resModel: "category",
        arch: `
            <kanban highlight_color="colorpickerField">
                <templates>
                    <t t-name="menu">
                        <field name="colorpickerField" widget="kanban_color_picker"/>
                    </t>
                    <t t-name="card">
                        <field name="name"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect(getKanbanRecord({ index: 0 })).toHaveClass("o_kanban_color_3");
    await toggleKanbanRecordDropdown(0);
    await contains(`.o_kanban_colorpicker .o_colorlist_item_color_9[title="Raspberry"]`).click();
    // should write on the color field
    expect.verifySteps(["write-color-9"]);
    expect(getKanbanRecord({ index: 0 })).toHaveClass("o_kanban_color_9");
});

test("edit the kanban color with translated colors resulting in the same terms", async () => {
    Category._records[0].color = 12;

    const translations = {
        Purple: "Violet",
        Violet: "Violet",
    };
    defineParams({ translations });

    await mountView({
        type: "kanban",
        resModel: "category",
        arch: `
            <kanban highlight_color="color">
                <templates>
                    <t t-name="menu">
                        <field name="color" widget="kanban_color_picker"/>
                    </t>
                    <t t-name="card">
                        <field name="name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await toggleKanbanRecordDropdown(0);
    await contains(".o_kanban_colorpicker .o_colorlist_item_color_9").click();
    expect(getKanbanRecord({ index: 0 })).toHaveClass("o_kanban_color_9");
});

test("colorpicker doesn't appear when missing access rights", async () => {
    await mountView({
        type: "kanban",
        resModel: "category",
        arch: `
            <kanban edit="0">
                <templates>
                    <t t-name="menu">
                        <field name="color" widget="kanban_color_picker"/>
                    </t>
                    <t t-name="card">
                        <field name="name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await toggleKanbanRecordDropdown(0);
    expect(".o_kanban_colorpicker").toHaveCount(0);
});

test("load more records in column", async () => {
    onRpc("web_search_read", ({ kwargs }) => {
        expect.step(`web_search_read ${kwargs.limit} - ${kwargs.offset}`);
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
        limit: 2,
    });

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2, {
        message: "there should be 2 records in the column",
    });
    expect(getKanbanRecordTexts(1)).toEqual(["1", "2"]);

    // load more
    await clickKanbanLoadMore(1);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(3, {
        message: "there should now be 3 records in the column",
    });
    // the records should be correctly fetched
    expect(getKanbanRecordTexts(1)).toEqual(["1", "2", "3"]);
    expect.verifySteps(["web_search_read 4 - 0"]);

    // reload
    await validateSearch();

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(3, {
        message: "there should still be 3 records in the column after reload",
    });
    expect(getKanbanRecordTexts(1)).toEqual(["1", "2", "3"]);
    expect.verifySteps([]); // managed by web_read_group
});

test("load more records in column with x2many", async () => {
    Partner._records[0].category_ids = [7];
    Partner._records[1].category_ids = [];
    Partner._records[2].category_ids = [6];
    Partner._records[3].category_ids = [];
    // record [2] will be loaded after

    onRpc("web_search_read", ({ kwargs }) => {
        expect.step(`web_search_read ${kwargs.limit} - ${kwargs.offset}`);
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="category_ids"/>
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
        limit: 2,
    });

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);
    expect(queryAllTexts("[name='category_ids']", { root: getKanbanColumn(1) })).toEqual([
        "silver",
        "",
    ]);

    // load more
    await clickKanbanLoadMore(1);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(3);
    expect(queryAllTexts("[name='category_ids']", { root: getKanbanColumn(1) })).toEqual([
        "silver",
        "",
        "gold",
    ]);
    expect.verifySteps(["web_search_read 4 - 0"]);
});

test("update buttons after column creation", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o-kanban-button-new").not.toBeEnabled();
    await editKanbanColumnName("new column");
    await validateKanbanColumn();
    expect(".o-kanban-button-new").toBeEnabled();
});

test.tags("desktop");
test("group_by_tooltip option when grouping on a many2one", async () => {
    Partner._records[3].product_id = false;

    onRpc("read", ({ args }) => {
        expect.step("read: product");
        expect(args[1]).toEqual(["display_name", "name"], {
            message: "should read on specified fields on the group by relation",
        });
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="bar">
                <field name="product_id" options='{"group_by_tooltip": {"name": "Kikou"}}'/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_product_id" domain="[]" string="GroupBy Product" context="{ 'group_by': 'product_id' }"/>
            </search>`,
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_kanban_group").toHaveCount(2, { message: "should have 2 columns" });

    // simulate an update coming from the searchview, with another groupby given
    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Product");

    expect(".o_kanban_group").toHaveCount(3, { message: "should have 3 columns" });
    expect(".o_kanban_group:first").toHaveClass("o_column_folded");

    await contains(".o_kanban_group").click();
    expect(".o_kanban_group").toHaveCount(3, { message: "should have 3 columns" });
    expect(".o_kanban_group:first").not.toHaveClass("o_column_folded");
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(1);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(2) })).toHaveCount(1);
    expect(queryText(".o_column_title", { root: getKanbanColumn(0) })).toBe("None\n(1)", {
        message: "first column should have a default title for when no value is provided",
    });

    await hover(".o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(0, {
        message:
            "tooltip of first column should not defined, since group_by_tooltip title and the many2one field has no value",
    });
    // should not have done any read on product because no value
    expect.verifySteps([]);

    await hover(".o_column_title:eq(1)");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(1, {
        message:
            "second column should have a tooltip with the group_by_tooltip title and many2one field value",
    });
    expect(".o-tooltip:first").toHaveText("Kikou\nhello");
    expect(".o_kanban_group:nth-child(2) .o_column_title").toHaveText("hello\n(2)", {
        message: "second column should have a title with a value from the many2one",
    });
    // should have done one read on product for the second column tooltip
    expect.verifySteps(["read: product"]);
});

test.tags("desktop");
test("asynchronous tooltips when grouped", async () => {
    const def = new Deferred();
    onRpc("read", () => {
        expect.step("read: product");
        return def;
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <field name="product_id" options='{"group_by_tooltip": {"name": "Name"}}'/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_column_title").toHaveCount(2);

    await hover(".o_kanban_group .o_kanban_header_title .o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(0);

    await leave();
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(0);

    await hover(".o_kanban_group .o_kanban_header_title .o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(0);

    def.resolve();
    await animationFrame();

    expect(".o-tooltip").toHaveCount(1);
    expect(".o-tooltip").toHaveText("Name\nhello");
    expect.verifySteps(["read: product"]);
});

test.tags("desktop");
test("loads data tooltips only when first opening", async () => {
    onRpc("read", () => {
        expect.step("read: product");
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <field name="product_id" options='{"group_by_tooltip": {"name": "Name"}}'/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await hover(".o_kanban_group .o_kanban_header_title .o_column_title");
    await await runAllTimers();
    expect(".o-tooltip").toHaveCount(1);
    expect(".o-tooltip").toHaveText("Name\nhello");
    expect.verifySteps(["read: product"]);

    await leave();
    await animationFrame();
    expect(".o-tooltip").toHaveCount(0, { message: "tooltip should be closed" });

    await hover(".o_kanban_group .o_kanban_header_title .o_column_title");
    await runAllTimers();
    expect(".o-tooltip").toHaveCount(1);
    expect(".o-tooltip").toHaveText("Name\nhello");
    expect.verifySteps([]);
});

test.tags("desktop");
test("move a record then put it again in the same column", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="display_name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await editKanbanColumnName("column1");
    await validateKanbanColumn();

    await editKanbanColumnName("column2");
    await validateKanbanColumn();

    await quickCreateKanbanRecord(1);
    await editKanbanRecordQuickCreateInput("display_name", "new partner");
    await validateKanbanRecord();

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);

    await contains(".o_kanban_group:nth-child(2) .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first-child"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(0);

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
});

test.tags("desktop");
test("resequence a record twice", async () => {
    Partner._records = [];

    const def = new Deferred();
    onRpc("web_resequence", () => {
        expect.step("resequence");
        return def;
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="display_name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await editKanbanColumnName("column1");
    await validateKanbanColumn();

    await quickCreateKanbanRecord();
    await editKanbanRecordQuickCreateInput("display_name", "record1");
    await validateKanbanRecord();

    await quickCreateKanbanRecord();
    await editKanbanRecordQuickCreateInput("display_name", "record2");
    await validateKanbanRecord();
    await discardKanbanRecord(); // close quick create

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(getKanbanRecordTexts()).toEqual(["record2", "record1"], {
        message: "records should be correctly ordered",
    });

    await contains(".o_kanban_record:nth-child(2)").dragAndDrop(".o_kanban_record:nth-child(3)");
    def.resolve();
    await animationFrame();

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(getKanbanRecordTexts()).toEqual(["record1", "record2"], {
        message: "records should be correctly ordered",
    });

    await contains(".o_kanban_record:nth-child(3)").dragAndDrop(".o_kanban_record:nth-child(2)");

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(getKanbanRecordTexts()).toEqual(["record2", "record1"], {
        message: "records should be correctly ordered",
    });
    // should have resequenced twice
    expect.verifySteps(["resequence", "resequence"]);
});

test("basic support for widgets (being Owl Components)", async () => {
    class MyComponent extends Component {
        static template = xml`<div t-att-class="props.class" t-esc="value"/>`;
        static props = ["*"];
        get value() {
            return JSON.stringify(this.props.record.data);
        }
    }
    const myComponent = {
        component: MyComponent,
    };
    viewWidgetRegistry.add("test", myComponent);
    after(() => viewWidgetRegistry.remove("test"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <widget name="test"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecord({ index: 2 }).querySelector(".o_widget")).toHaveText('{"foo":"gnap"}');
});

test("kanban card: record value should be updated", async () => {
    class MyComponent extends Component {
        static template = xml`<div><button t-on-click="onClick">CLick</button></div>`;
        static props = ["*"];
        onClick() {
            this.props.record.update({ foo: "yolo" });
        }
    }
    const myComponent = {
        component: MyComponent,
    };
    viewWidgetRegistry.add("test", myComponent);
    after(() => viewWidgetRegistry.remove("test"));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" class="foo"/>
                        <widget name="test"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(queryText(".foo", { root: getKanbanRecord({ index: 0 }) })).toBe("yop");

    await click(queryOne("button", { root: getKanbanRecord({ index: 0 }) }));
    await animationFrame();
    await animationFrame();

    expect(queryText(".foo", { root: getKanbanRecord({ index: 0 }) })).toBe("yolo");
});

test.tags("desktop");
test("load more should load correct records after drag&drop event", async () => {
    Partner._order = ["sequence", "id"];
    Partner._records.forEach((r, i) => (r.sequence = i));

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban limit="1">
                <templates>
                    <t t-name="card">
                        <field name="id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });

    expect(getKanbanRecordTexts(0)).toEqual(["4"]);
    expect(getKanbanRecordTexts(1)).toEqual(["1"]);

    // Drag the first kanban record on top of the last
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:last-child .o_kanban_record"
    );

    // load more twice to load all records of second column
    await clickKanbanLoadMore(1);
    await clickKanbanLoadMore(1);

    // Check records of the second column
    expect(getKanbanRecordTexts(1)).toEqual(["4", "1", "2", "3"]);
});

test.tags("desktop");
test("grouped kanban: clear groupby when reloading", async () => {
    // in this test, we simulate that clearing the domain is slow, so that
    // clearing the groupby does not corrupt the data handled while
    // reloading the kanban view.
    const def = new Deferred();
    onRpc("web_read_group", async function ({ kwargs, parent }) {
        const result = parent();
        if (kwargs.domain.length === 0 && kwargs.groupby && kwargs.groupby[0] === "bar") {
            await def; // delay 1st update
        }
        return result;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="my_filter" string="My Filter" domain="[['foo', '=', 'norecord']]"/>
                <filter name="group_by_bar" domain="[]" string="GroupBy Bar" context="{ 'group_by': 'bar' }"/>
            </search>`,
        context: {
            search_default_group_by_bar: 1,
            search_default_my_filter: 1,
        },
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_kanban_renderer").not.toHaveClass("o_kanban_ungrouped");
    expect(queryAllTexts(".o_facet_value")).toEqual(["My Filter", "GroupBy Bar"]);

    await contains(".o_facet_remove:first").click();
    await contains(".o_facet_remove:only").click();
    def.resolve(); // simulate slow 1st update of kanban view
    await animationFrame();

    expect(".o_kanban_renderer").not.toHaveClass("o_kanban_grouped");
    expect(".o_kanban_renderer").toHaveClass("o_kanban_ungrouped");
});

test("keynav: right/left", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await pointerDown(getKanbanRecord({ index: 0 }));
    expect(getKanbanRecord({ index: 0 })).toBeFocused();

    await press("ArrowRight");
    expect(getKanbanRecord({ index: 1 })).toBeFocused();

    await press("ArrowLeft");
    expect(getKanbanRecord({ index: 0 })).toBeFocused();
});

test("keynav: down, with focus is inside a card", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <a href="#" class="o-this-is-focussable">ho! this is focussable</a>
                    </t>
                </templates>
            </kanban>`,
    });

    await pointerDown(getKanbanRecord({ index: 0 }).querySelector(".o-this-is-focussable"));
    await press("ArrowDown");

    expect(getKanbanRecord({ index: 1 })).toBeFocused();
});

test.tags("desktop");
test("keynav: grouped kanban", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
    });
    const cardsByColumn = queryAll(".o_kanban_group").map((root) =>
        queryAll(".o_kanban_record", { root })
    );
    const firstColumnFirstCard = cardsByColumn[0][0];
    const secondColumnFirstCard = cardsByColumn[1][0];
    const secondColumnSecondCard = cardsByColumn[1][1];

    // DOWN should focus the first card
    await press("ArrowDown");
    expect(firstColumnFirstCard).toBeFocused({
        message: "LEFT should select the first card of the first column",
    });

    // RIGHT should select the next column
    await press("ArrowRight");
    expect(secondColumnFirstCard).toBeFocused({
        message: "RIGHT should select the first card of the next column",
    });

    // DOWN should move up one card
    await press("ArrowDown");
    expect(secondColumnSecondCard).toBeFocused({
        message: "DOWN should select the second card of the current column",
    });

    // LEFT should go back to the first column
    await press("ArrowLeft");
    expect(firstColumnFirstCard).toBeFocused({
        message: "LEFT should select the first card of the first column",
    });
});

test.tags("desktop");
test("keynav: grouped kanban with empty columns", async () => {
    Partner._records[1].state = "abc";

    onRpc("web_read_group", function ({ parent }) {
        // override web_read_group to return empty groups, as this is
        // the case for several models (e.g. project.task grouped
        // by stage_id)
        const result = parent();
        // add 2 empty columns in the middle
        result.groups.splice(1, 0, {
            __count: 0,
            state: "md1",
            __extra_domain: [["state", "=", "md1"]],
        });
        result.groups.splice(1, 0, {
            __count: 0,
            state: "md2",
            __extra_domain: [["state", "=", "md2"]],
        });
        // add 1 empty column in the beginning and the end
        result.groups.unshift({
            __count: 0,
            state: "beg",
            __extra_domain: [["state", "=", "beg"]],
        });
        result.groups.push({
            __count: 0,
            state: "end",
            __extra_domain: [["state", "=", "end"]],
        });
        return result;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["state"],
    });

    /**
     * Added columns in mockRPC are empty
     *
     *    | BEG | ABC  | MD1 | MD2 | GHI  | END
     *    |-----|------|-----|-----|------|-----
     *    |     | yop  |     |     | gnap |
     *    |     | blip |     |     | blip |
     */
    const cardsByColumn = queryAll(".o_kanban_group").map((root) =>
        queryAll(".o_kanban_record", { root })
    );
    const yop = cardsByColumn[1][0];
    const gnap = cardsByColumn[4][0];

    // DOWN should focus yop (first card)
    await press("ArrowDown");
    expect(yop).toBeFocused({
        message: "LEFT should select the first card of the first column that has a card",
    });

    // RIGHT should select the next column that has a card
    await press("ArrowRight");
    expect(gnap).toBeFocused({
        message: "RIGHT should select the first card of the next column that has a card",
    });

    // LEFT should go back to the first column that has a card
    await press("ArrowLeft");
    expect(yop).toBeFocused({
        message: "LEFT should select the first card of the first column that has a card",
    });
});

test.tags("desktop");
test("keynav: no global_click, press ENTER on card with a link", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban can_open="0">
                <templates>
                    <t t-name="card">
                        <a type="archive">Archive</a>
                    </t>
                </templates>
            </kanban>`,
        selectRecord: (resId) => {
            expect.step("select record");
        },
    });

    await press("ArrowDown");
    expect(".o_kanban_record:first").toBeFocused();
    await press("Enter");

    await animationFrame();
    expect(".o_dialog").toHaveCount(1);
    expect(".o_dialog main").toHaveText("Are you sure that you want to archive this record?");
    expect.verifySteps([]); // should not try to open the record
});

test.tags("desktop");
test("keynav: kanban with global_click", async () => {
    expect.assertions(2);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <a name="action_test" type="object" />
                    </t>
                </templates>
            </kanban>`,
        selectRecord(recordId) {
            expect(recordId).toBe(1, {
                message: "should call its selectRecord prop with the selected record",
            });
        },
    });

    await press("ArrowDown");
    expect(".o_kanban_record:first").toBeFocused();
    await press("Enter");
});

test.tags("desktop");
test(`kanban should ask to scroll to top on page changes`, async () => {
    // add records to be able to scroll
    for (let i = 5; i < 200; i++) {
        Partner._records.push({ id: i, foo: "foo" });
    }
    patchWithCleanup(KanbanController.prototype, {
        onPageChangeScroll() {
            super.onPageChangeScroll(...arguments);
            expect.step("scroll");
        },
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    // switch pages (should ask to scroll)
    await pagerNext();
    await pagerPrevious();
    // should ask to scroll when switching pages
    expect.verifySteps(["scroll", "scroll"]);

    // change the limit (should not ask to scroll)
    await contains(`.o_pager_value`).click();
    await contains(`.o_pager_value`).edit("1-100");
    await animationFrame();
    expect(getPagerValue()).toEqual([1, 100]);
    // should not ask to scroll when changing the limit
    expect.verifySteps([]);

    await contains(".o_content").scroll({ top: 250 });
    expect(".o_content").toHaveProperty("scrollTop", 250);

    // switch pages again (should still ask to scroll)
    await pagerNext();
    // this is still working after a limit change
    expect.verifySteps(["scroll"]);
    // Should effectively reset the scroll position
    expect(".o_content").toHaveProperty("scrollTop", 0);
});

test.tags("mobile");
test(`kanban should ask to scroll to top on page changes (mobile)`, async () => {
    // add records to be able to scroll
    for (let i = 5; i < 200; i++) {
        Partner._records.push({ id: i, foo: "foo" });
    }
    patchWithCleanup(KanbanController.prototype, {
        onPageChangeScroll() {
            super.onPageChangeScroll(...arguments);
            expect.step("scroll");
        },
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    // switch pages (should ask to scroll)
    await pagerNext();
    await pagerPrevious();
    // should ask to scroll when switching pages
    expect.verifySteps(["scroll", "scroll"]);

    await contains(".o_kanban_view").scroll({ top: 250 });
    expect(".o_kanban_view").toHaveProperty("scrollTop", 250);

    // switch pages again (should still ask to scroll)
    await pagerNext();
    expect.verifySteps(["scroll"]);
    // Should effectively reset the scroll position
    expect(".o_kanban_view").toHaveProperty("scrollTop", 0);
});

test.tags("desktop");
test("set cover image", async () => {
    expect.assertions(8);

    IrAttachment._records = [
        {
            id: 1,
            name: "1.png",
            mimetype: "image/png",
            res_model: "partner",
            res_id: 1,
            create_uid: false,
        },
        {
            id: 2,
            name: "2.png",
            mimetype: "image/png",
            res_model: "partner",
            res_id: 2,
            create_uid: false,
        },
    ];
    Partner._fields.displayed_image_id = fields.Many2one({
        string: "Cover",
        relation: "ir.attachment",
    });

    onRpc("partner", "web_save", ({ args }) => {
        expect.step(args[0][0]);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="menu">
                        <a type="set_cover" data-field="displayed_image_id" class="dropdown-item">Set Cover Image</a>
                    </t>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="displayed_image_id" widget="attachment_image"/>
                    </t>
                </templates>
            </kanban>`,
    });

    mockService("action", {
        switchView(_viewType, { readonly, resModel, res_id, view_type }) {
            expect({ readonly, resModel, res_id, view_type }).toBe({
                readonly: true,
                resModel: "partner",
                res_id: 1,
                view_type: "form",
            });
        },
    });

    await toggleKanbanRecordDropdown(0);
    await contains(".oe_kanban_action", {
        root: getDropdownMenu(getKanbanRecord({ index: 0 })),
    }).click();

    expect(queryAll("img", { root: getKanbanRecord({ index: 0 }) })).toHaveCount(0, {
        message: "Initially there is no image.",
    });

    // The image is immediately assigned on click
    await contains(".modal .o_kanban_cover_image img").click();

    expect('img[data-src*="/web/image/1"]').toHaveCount(1);

    await toggleKanbanRecordDropdown(1);
    const coverButton = getDropdownMenu(getKanbanRecord({ index: 1 })).querySelector("a");
    expect(queryText(coverButton)).toBe("Set Cover Image");
    await contains(coverButton).click();

    expect(".modal .o_kanban_cover_image").toHaveCount(1);
    expect(".modal .btn:contains(Discard)").toHaveCount(1);
    expect(".modal .btn:contains(Remove Cover)").toHaveCount(0);

    await contains(".modal .o_kanban_cover_image img").click(); // Assign the image as cover in one click
    await animationFrame();

    expect('img[data-src*="/web/image/2"]').toHaveCount(1);

    await contains(".o_kanban_record:first-child .o_attachment_image").click(); //Not sure, to discuss

    // should writes on both kanban records
    expect.verifySteps([1, 2]);
});

test.tags("desktop");
test("open file explorer if no cover image", async () => {
    expect.assertions(2);

    Partner._fields.displayed_image_id = fields.Many2one({
        string: "Cover",
        relation: "ir.attachment",
    });

    const uploadedPromise = new Deferred();
    await createFileInput({
        mockPost: async (route) => {
            if (route === "/web/binary/upload_attachment") {
                await uploadedPromise;
            }
            return "[]";
        },
        props: {},
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="menu">
                        <a type="set_cover" data-field="displayed_image_id" class="dropdown-item">Set Cover Image</a>
                    </t>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="displayed_image_id" widget="attachment_image"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await toggleKanbanRecordDropdown(0);
    await contains(".oe_kanban_action", {
        root: getDropdownMenu(getKanbanRecord({ index: 0 })),
    }).click();
    await setInputFiles([]);
    await animationFrame();

    expect(`.o_dialog .o_file_input input`).not.toBeEnabled({
        message: "the upload button should be disabled on upload",
    });
    uploadedPromise.resolve();
    await animationFrame();

    expect(`.o_dialog .o_file_input input`).toBeEnabled({
        message: "the upload button should be enabled for upload",
    });
});

test.tags("desktop");
test("unset cover image", async () => {
    IrAttachment._records = [
        {
            id: 1,
            name: "1.png",
            mimetype: "image/png",
            res_model: "partner",
            res_id: 1,
            create_uid: false,
        },
        {
            id: 2,
            name: "2.png",
            mimetype: "image/png",
            res_model: "partner",
            res_id: 2,
            create_uid: false,
        },
    ];
    Partner._fields.displayed_image_id = fields.Many2one({
        string: "Cover",
        relation: "ir.attachment",
    });
    Partner._records[0].displayed_image_id = 1;
    Partner._records[1].displayed_image_id = 2;

    onRpc("partner", "web_save", ({ args }) => {
        expect.step(args[0][0]);
        expect(args[1].displayed_image_id).toBe(false);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="menu">
                        <a type="set_cover" data-field="displayed_image_id" class="dropdown-item">Set Cover Image</a>
                    </t>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="displayed_image_id" widget="attachment_image"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await toggleKanbanRecordDropdown(0);
    await contains(".oe_kanban_action", {
        root: getDropdownMenu(getKanbanRecord({ index: 0 })),
    }).click();

    expect(
        queryAll('img[data-src*="/web/image/1"]', { root: getKanbanRecord({ index: 0 }) })
    ).toHaveCount(1);
    expect(
        queryAll('img[data-src*="/web/image/2"]', { root: getKanbanRecord({ index: 1 }) })
    ).toHaveCount(1);

    expect(".modal .o_kanban_cover_image").toHaveCount(1);
    expect(".modal .btn:contains(Discard)").toHaveCount(1);
    expect(".modal .btn:contains(Remove Cover)").toHaveCount(1);

    await contains(".modal .btn-danger").click(); // click on "Remove Cover" button

    expect(queryAll("img", { root: getKanbanRecord({ index: 0 }) })).toHaveCount(0, {
        message: "The cover image should be removed.",
    });

    await toggleKanbanRecordDropdown(1);
    const coverButton = getDropdownMenu(getKanbanRecord({ index: 1 })).querySelector("a");
    expect(queryText(coverButton)).toBe("Set Cover Image");
    await contains(coverButton).click();

    await contains(".modal .o_kanban_cover_image img").click(); // Assign the image as cover in one click
    await animationFrame();

    expect(queryAll("img", { root: getKanbanRecord({ index: 1 }) })).toHaveCount(0, {
        message: "The cover image should be removed.",
    });
    // should writes on both kanban records
    expect.verifySteps([1, 2]);
});

test.tags("desktop");
test("ungrouped kanban with handle field", async () => {
    expect.assertions(4);

    onRpc("web_search_read", ({ kwargs }) => {
        expect.step(`web_search_read: order: ${kwargs.order}`);
    });
    onRpc("web_resequence", ({ args }) => {
        expect(args[0]).toEqual([2, 1, 3, 4], {
            message: "should write the sequence in correct order",
        });
        return [];
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="int_field" widget="handle" />
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual(["blip", "blip", "yop", "gnap"]);

    await contains(".o_kanban_record").dragAndDrop(queryFirst(".o_kanban_record:nth-child(4)"));

    expect(getKanbanRecordTexts()).toEqual(["blip", "yop", "gnap", "blip"]);
    expect.verifySteps(["web_search_read: order: int_field ASC, id ASC"]);
});

test("ungrouped kanban without handle field", async () => {
    onRpc("web_resequence", () => {
        expect.step("resequence");
        return [];
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual(["yop", "blip", "gnap", "blip"]);

    await contains(".o_kanban_record").dragAndDrop(queryFirst(".o_kanban_record:nth-child(4)"));

    expect(getKanbanRecordTexts()).toEqual(["yop", "blip", "gnap", "blip"]);
    expect.verifySteps([]);
});

test("click on image field in kanban (with default global_click)", async () => {
    expect.assertions(2);

    Partner._fields.image = fields.Binary();
    Partner._records[0].image = "R0lGODlhAQABAAD/ACwAAAAAAQABAAACAA==";

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="image" widget="image"/>
                    </t>
                </templates>
            </kanban>`,
        selectRecord(recordId) {
            expect(recordId).toBe(1, {
                message: "should call its selectRecord prop with the clicked record",
            });
        },
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    await contains(".o_field_image").click();
});

test("kanban view with boolean field", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="bar"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record input:disabled").toHaveCount(4);
    expect(".o_kanban_record input:checked").toHaveCount(3);
    expect(".o_kanban_record input:not(:checked)").toHaveCount(1);
});

test("kanban view with boolean widget", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="bar" widget="boolean"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(
        queryAll("div.o_field_boolean .o-checkbox", { root: getKanbanRecord({ index: 0 }) })
    ).toHaveCount(1);
});

test("kanban view with boolean toggle widget", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="bar" widget="boolean_toggle"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect(getKanbanRecord({ index: 0 }).querySelector("[name='bar'] input")).toBeChecked();
    expect(getKanbanRecord({ index: 1 }).querySelector("[name='bar'] input")).toBeChecked();

    await click("[name='bar'] input:only", { root: getKanbanRecord({ index: 1 }) });
    await animationFrame();

    expect(getKanbanRecord({ index: 0 }).querySelector("[name='bar'] input")).toBeChecked();
    expect(getKanbanRecord({ index: 1 }).querySelector("[name='bar'] input")).not.toBeChecked();
});

test("kanban view with monetary and currency fields without widget", async () => {
    const mockedCurrencies = {};
    for (const record of Currency._records) {
        mockedCurrencies[record.id] = record;
    }
    patchWithCleanup(currencies, mockedCurrencies);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="currency_id"/>
                <templates>
                    <t t-name="card">
                        <field name="salary"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual([
        `$ 1,750.00`,
        `$ 1,500.00`,
        `2,000.00 €`,
        `$ 2,222.00`,
    ]);
});

test("kanban widget can extract props from attrs", async () => {
    class TestWidget extends Component {
        static template = xml`<div class="o-test-widget-option" t-esc="props.title"/>`;
        static props = ["*"];
    }
    const testWidget = {
        component: TestWidget,
        extractProps: ({ attrs }) => ({
            title: attrs.title,
        }),
    };
    viewWidgetRegistry.add("widget_test_option", testWidget);
    after(() => viewWidgetRegistry.remove("widget_test_option"));

    await mountView({
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <widget name="widget_test_option" title="Widget with Option"/>
                    </t>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
    });

    expect(".o-test-widget-option").toHaveCount(4);
    expect(".o-test-widget-option:first").toHaveText("Widget with Option");
});

test("action/type attributes on kanban arch, type='object'", async () => {
    mockService("action", {
        doActionButton(params) {
            expect.step(`doActionButton type ${params.type} name ${params.name}`);
            params.onClose();
        },
    });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban action="a1" type="object">
                <templates>
                    <t t-name="card">
                        <p>some value</p><field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);
    await contains(".o_kanban_record p").click();
    expect.verifySteps(["doActionButton type object name a1", "web_search_read"]);
});

test("action/type attributes on kanban arch, type='action'", async () => {
    mockService("action", {
        doActionButton(params) {
            expect.step(`doActionButton type ${params.type} name ${params.name}`);
            params.onClose();
        },
    });

    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban action="a1" type="action">
                <templates>
                    <t t-name="card">
                        <p>some value</p><field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);
    await contains(".o_kanban_record p").click();
    expect.verifySteps(["doActionButton type action name a1", "web_search_read"]);
});

test("Missing t-key is automatically filled with a warning", async () => {
    patchWithCleanup(console, { warn: () => expect.step("warning") });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <div>
                            <span t-foreach="[1, 2, 3]" t-as="i" t-esc="i" />
                        </div>
                    </t>
                </templates>
            </kanban>`,
    });

    expect.verifySteps(["warning"]);
    expect(getKanbanRecord({ index: 0 })).toHaveText("123");
});

test("Allow use of 'editable'/'deletable' in ungrouped kanban", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <button t-if="widget.editable">EDIT</button>
                        <button t-if="widget.deletable">DELETE</button>
                    </div>
                </templates>
            </kanban>`,
    });

    expect(getKanbanRecordTexts()).toEqual([
        "EDITDELETE",
        "EDITDELETE",
        "EDITDELETE",
        "EDITDELETE",
    ]);
});

test.tags("desktop");
test("folded groups kept when leaving/coming back", async () => {
    Partner._views = {
        kanban: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                    </t>
                </templates>
            </kanban>`,
    };
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        name: "Partners",
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_column_folded").toHaveCount(0);
    expect(".o_kanban_record").toHaveCount(4);

    // fold the first group
    const clickColumnAction = await toggleKanbanColumnActions(0);
    await clickColumnAction("Fold");
    expect(".o_column_folded").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(2);

    // open a record and go back
    await contains(".o_kanban_record").click();
    expect(".o_form_view").toHaveCount(1);

    await contains(".breadcrumb-item a").click();
    expect(".o_column_folded").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(2);
});

test.tags("desktop");
test("folded groups kept when leaving/coming back (grouped by date)", async () => {
    Partner._fields.date = fields.Date({ default: "2022-10-10" });
    Partner._records[0].date = "2022-05-10";
    Partner._views = {
        kanban: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                    </t>
                </templates>
            </kanban>`,
    };
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        name: "Partners",
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
        context: {
            group_by: ["date"],
        },
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_column_folded").toHaveCount(0);
    expect(".o_kanban_record").toHaveCount(4);

    // fold the second column
    const clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Fold");
    expect(".o_column_folded").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(1);

    // open a record and go back
    await contains(".o_kanban_record").click();
    expect(".o_form_view").toHaveCount(1);

    await contains(".breadcrumb-item a").click();
    expect(".o_column_folded").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(1);
});

test.tags("desktop");
test("loaded records kept when leaving/coming back", async () => {
    Partner._views = {
        kanban: `
            <kanban limit="1">
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                    </t>
                </templates>
            </kanban>`,
    };
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        name: "Partners",
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_record").toHaveCount(2);

    // load more records in second group
    await clickKanbanLoadMore(1);
    expect(".o_kanban_record").toHaveCount(3);

    // open a record and go back
    await contains(".o_kanban_record").click();
    expect(".o_form_view").toHaveCount(1);

    await contains(".breadcrumb-item a").click();
    expect(".o_kanban_record").toHaveCount(3);
});

test("basic rendering with 2 groupbys", async () => {
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar", "product_id"],
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
    ]);
});

test("basic rendering with a date groupby with a granularity", async () => {
    Partner._records[0].date = "2022-06-23";

    stepAllNetworkCalls();
    onRpc("web_read_group", ({ method, kwargs }) => {
        expect(kwargs.aggregates).toEqual([]);
        expect(kwargs.groupby).toEqual(["date:day"]);
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["date:day"],
    });

    expect(".o_kanban_renderer").toHaveClass("o_kanban_grouped");
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(3);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
    ]);
});

test("dropdown is closed on item click", async () => {
    Partner._records.splice(1, 3); // keep one record only

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="menu">
                        <a role="menuitem" class="dropdown-item">Item</a>
                    </t>
                    <t t-name="card">
                        <div/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o-dropdown--menu").toHaveCount(0);

    await toggleKanbanRecordDropdown();

    expect(".o-dropdown--menu").toHaveCount(1);

    await contains(".o-dropdown--menu .dropdown-item").click();

    expect(".o-dropdown--menu").toHaveCount(0);
});

test("can use JSON in kanban template", async () => {
    Partner._records = [{ id: 1, foo: '["g", "e", "d"]' }];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <field name="foo"/>
                <templates>
                    <t t-name="card">
                        <div>
                            <span t-foreach="JSON.parse(record.foo.raw_value)" t-as="v" t-key="v_index" t-esc="v"/>
                        </div>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(1);
    expect(".o_kanban_record span").toHaveCount(3);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveText("ged");
});

test.tags("desktop");
test("keep focus in cp when pressing arrowdown and no kanban card", async () => {
    Partner._records = [];

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="display_name"/>
                    </t>
                </templates>
            </kanban>`,
    });

    // Check that there is a column quick create
    expect(".o_column_quick_create").toHaveCount(1);
    await editKanbanColumnName("new col");
    await validateKanbanColumn();

    // Check that there is only one group and no kanban card
    expect(".o_kanban_group").toHaveCount(1);
    expect(".o_kanban_group.o_kanban_no_records").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(0);

    // Check that the focus is on the searchview input
    await quickCreateKanbanRecord();
    expect(".o_kanban_group.o_kanban_no_records").toHaveCount(1);
    expect(".o_kanban_quick_create").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(0);

    // Somehow give the focus in the control panel, i.e. in the search view
    // Note that a simple click in the control panel should normally close the quick
    // create, so in order to give the focus in the search input, the user would
    // normally have to right-click on it then press escape. These are behaviors
    // handled through the browser, so we simply call focus directly here.
    queryFirst(".o_searchview_input").focus();

    // Make sure no async code will have a side effect on the focused element
    await animationFrame();
    expect(".o_searchview_input").toBeFocused();

    // Trigger the ArrowDown hotkey
    await press("ArrowDown");
    await animationFrame();
    expect(".o_searchview_input").toBeFocused();
});

test.tags("desktop");
test("no leak of TransactionInProgress (grouped case)", async () => {
    const def = new Deferred();
    onRpc("web_resequence", () => {
        expect.step("resequence");
        return def;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["state"],
    });

    expect(".o_kanban_group:nth-child(1) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(1) .o_kanban_record").toHaveText("yop");
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveText("blip");
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);

    expect.verifySteps([]);

    // move "yop" from first to second column
    await contains(".o_kanban_group:nth-child(1) .o_kanban_record").dragAndDrop(
        queryFirst(".o_kanban_group:nth-child(2)")
    );

    expect(".o_kanban_group:nth-child(1) .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(queryAllTexts(".o_kanban_group:nth-child(2) .o_kanban_record")).toEqual(["blip", "yop"]);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);
    expect.verifySteps(["resequence"]);

    // try to move "yop" from second to third column
    await contains(".o_kanban_group:nth-child(2) .o_kanban_record:nth-child(3)").dragAndDrop(
        ".o_kanban_group:nth-child(3)"
    );

    expect(".o_kanban_group:nth-child(1) .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
    expect(queryAllTexts(".o_kanban_group:nth-child(2) .o_kanban_record")).toEqual(["blip", "yop"]);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);
    expect.verifySteps([]);

    def.resolve();
    await animationFrame();

    // try again to move "yop" from second to third column
    await contains(".o_kanban_group:nth-child(2) .o_kanban_record:nth-child(3)").dragAndDrop(
        ".o_kanban_group:nth-child(3)"
    );

    expect(".o_kanban_group:nth-child(1) .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(3);
    expect(queryAllTexts(".o_kanban_group:nth-child(3) .o_kanban_record")).toEqual([
        "gnap",
        "blip",
        "yop",
    ]);
    expect.verifySteps(["resequence"]);
});

test.tags("desktop");
test("no leak of TransactionInProgress (not grouped case)", async () => {
    const def = new Deferred();
    onRpc("web_resequence", () => {
        expect.step("resequence");
        return def;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban records_draggable="1">
                <field name="int_field" widget="handle" />
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "blip",
        "blip",
        "yop",
        "gnap",
    ]);
    expect.verifySteps([]);

    // move second "blip" to third place
    await contains(".o_kanban_record:nth-child(2)").dragAndDrop(".o_kanban_record:nth-child(3)");

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "blip",
        "yop",
        "blip",
        "gnap",
    ]);
    expect.verifySteps(["resequence"]);

    // try again
    await contains(".o_kanban_record:nth-child(2)").dragAndDrop(".o_kanban_record:nth-child(3)");
    expect.verifySteps([]);

    def.resolve();
    await animationFrame();

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "blip",
        "yop",
        "blip",
        "gnap",
    ]);

    await contains(".o_kanban_record:nth-child(3)").dragAndDrop(".o_kanban_record:nth-child(4)");

    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual([
        "blip",
        "yop",
        "gnap",
        "blip",
    ]);
    expect.verifySteps(["resequence"]);
});

test("fieldDependencies support for fields", async () => {
    const customField = {
        component: class CustomField extends Component {
            static template = xml`<span t-esc="props.record.data.int_field"/>`;
            static props = ["*"];
        },
        fieldDependencies: [{ name: "int_field", type: "integer" }],
    };
    fieldRegistry.add("custom_field", customField);
    after(() => fieldRegistry.remove("custom_field"));

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" widget="custom_field"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect("[name=foo] span:first").toHaveText("10");
});

test("fieldDependencies support for fields: dependence on a relational field", async () => {
    const customField = {
        component: class CustomField extends Component {
            static template = xml`<span t-esc="props.record.data.product_id.display_name"/>`;
            static props = ["*"];
        },
        fieldDependencies: [{ name: "product_id", type: "many2one", relation: "product" }],
    };
    fieldRegistry.add("custom_field", customField);
    after(() => fieldRegistry.remove("custom_field"));

    stepAllNetworkCalls();

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" widget="custom_field"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect("[name=foo] span:first").toHaveText("hello");
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_search_read",
        "has_group",
    ]);
});

test("column quick create - title and placeholder", async function (assert) {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o_column_quick_create.o_quick_create_folded").toHaveProperty(
        "textContent",
        " Add Product"
    );

    await quickCreateKanbanColumn();

    expect(
        ".o_column_quick_create.o_quick_create_unfolded .input-group .form-control"
    ).toHaveAttribute("placeholder", "Product...");
});

test.tags("desktop");
test("fold a column and drag record on it should not unfold it", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);

    const clickColumnAction = await toggleKanbanColumnActions(1);
    clickColumnAction("Fold");
    await animationFrame();

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(getKanbanColumn(1)).toHaveText("xmo\n(2)");

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(".o_column_folded");

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(1);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(getKanbanColumn(1)).toHaveText("xmo\n(3)");
});

test.tags("desktop");
test("drag record on initially folded column should not unfold it", async () => {
    Product._records[1].fold = true;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(queryText(getKanbanColumn(1))).toBe("xmo\n(2)");

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(".o_column_folded");

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(1);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(queryText(getKanbanColumn(1))).toBe("xmo\n(3)");
});

test.tags("desktop");
test("no sample data when all groups are folded then one is unfolded", async () => {
    Product._records.forEach((group) => {
        group.fold = true;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_column_folded").toHaveCount(2);

    await contains(".o_kanban_group").click();

    expect(".o_column_folded").toHaveCount(1);
    expect(".o_kanban_record").toHaveCount(2);
    expect("o_view_sample_data").toHaveCount(0);
});

test.tags("desktop");
test("no content helper, all groups folded with (unloaded) records", async () => {
    Product._records.forEach((group) => {
        group.fold = true;
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_column_folded").toHaveCount(2);
    expect(queryAllTexts(".o_column_title")).toEqual(["hello\n(2)", "xmo\n(2)"]);
    expect(".o_nocontent_help").toHaveCount(0);
});

test.tags("desktop");
test("Move multiple records in different columns simultaneously", async () => {
    const def = new Deferred();
    onRpc("read", () => def);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="id" />
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["state"],
    });

    expect(getKanbanRecordTexts()).toEqual(["1", "2", "3", "4"]);

    // Move 3 at end of 1st column
    await contains(".o_kanban_group:last-of-type .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first"
    );

    expect(getKanbanRecordTexts()).toEqual(["1", "3", "2", "4"]);

    // Move 4 at end of 1st column
    await contains(".o_kanban_group:last-of-type .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first"
    );

    expect(getKanbanRecordTexts()).toEqual(["1", "3", "4", "2"]);

    def.resolve();
    await animationFrame();

    expect(getKanbanRecordTexts()).toEqual(["1", "3", "4", "2"]);
});

test.tags("desktop");
test("drag & drop: content scrolls when reaching the edges", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="id" />
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["state"],
    });

    const width = 600;
    const content = queryOne(".o_content");
    content.setAttribute("style", `max-width:${width}px;overflow:auto;`);

    expect(content.scrollLeft).toBe(0);
    expect(content.getBoundingClientRect().width).toBe(600);
    expect(".o_kanban_record.o_dragged").toHaveCount(0);

    // Drag first record of first group to the right
    let dragActions = await contains(".o_kanban_record").drag();
    await dragActions.moveTo(".o_kanban_group:nth-child(3) .o_kanban_record:first");

    expect(".o_kanban_record.o_dragged").toHaveCount(1);

    // wait 30 frames, should be enough (default kanban speed is 20px per tick)
    await advanceFrame(30);

    // Should be at the end of the content
    expect(content.scrollLeft + width).toBe(content.scrollWidth);

    // Cancel drag: press "Escape"
    await press("Escape");
    await animationFrame();

    expect(".o_kanban_record.o_dragged").toHaveCount(0);

    // Drag first record of last group to the left
    dragActions = await contains(".o_kanban_group:nth-child(3) .o_kanban_record").drag();
    await dragActions.moveTo(".o_kanban_record:first");

    expect(".o_kanban_record.o_dragged").toHaveCount(1);

    await advanceFrame(30);

    expect(content.scrollLeft).toBe(0);

    // Cancel drag: click outside
    await contains(".o_kanban_renderer").click();

    expect(".o_kanban_record.o_dragged").toHaveCount(0);
});

test("attribute default_order", async () => {
    class CustomModel extends models.Model {
        _name = "custom.model";

        int = fields.Integer();

        _records = [
            { id: 1, int: 1 },
            { id: 2, int: 3 },
            { id: 3, int: 2 },
        ];
    }
    defineModels([CustomModel]);

    await mountView({
        type: "kanban",
        resModel: "custom.model",
        arch: `
            <kanban default_order="int">
                <templates>
                    <div t-name="card">
                        <field name="int" />
                    </div>
                </templates>
            </kanban>`,
    });
    expect(queryAllTexts(".o_kanban_record:not(.o_kanban_ghost)")).toEqual(["1", "2", "3"]);
});

test.tags("desktop");
test("d&d records grouped by m2o with m2o displayed in records", async () => {
    const readIds = [[2], [1, 3, 2]];
    const def = new Deferred();
    onRpc("read", ({ method, args }) => {
        expect(args[0]).toEqual(readIds[1]);
        return def;
    });
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="product_id" widget="many2one"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
    ]);
    expect(queryAllTexts(".o_kanban_record")).toEqual(["hello", "hello", "xmo", "xmo"]);

    await contains(".o_kanban_group:nth-child(2) .o_kanban_record").dragAndDrop(
        ".o_kanban_group:first-child"
    );
    expect(queryAllTexts(".o_kanban_record")).toEqual(["hello", "hello", "hello", "xmo"]);

    def.resolve();
    await animationFrame();

    expect.verifySteps(["web_save", "web_resequence"]);
    expect(queryAllTexts(".o_kanban_record")).toEqual(["hello", "hello", "hello", "xmo"]);
});

test("Can't use KanbanRecord implementation details in arch", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <div>
                            <t t-esc="__owl__"/>
                            <t t-esc="props"/>
                            <t t-esc="env"/>
                            <t t-esc="render"/>
                        </div>
                    </t>
                </templates>
            </kanban>`,
    });
    expect(".o_kanban_record:first").toHaveInnerHTML("<div></div>");
});

test.tags("desktop");
test("rerenders only once after resequencing records", async () => {
    // Actually it's not once, because we must render directly after the drag&drop s.t. the dropped
    // record remains where it has been dropped, once again after saving/reloading the record as
    // we rebuild record.data, and finally after the call to resequence, to re-enable the resequence
    // feature on the record (canResequence props).
    let saveDef = new Deferred();
    let resequenceDef = new Deferred();
    const renderCounts = {};
    patchWithCleanup(KanbanRecord.prototype, {
        setup() {
            super.setup();
            onWillRender(() => {
                const id = this.props.record.resId;
                renderCounts[id] = renderCounts[id] || 0;
                renderCounts[id]++;
            });
        },
    });

    onRpc("web_save", () => saveDef);
    onRpc("web_resequence", () => resequenceDef);
    stepAllNetworkCalls();

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(renderCounts).toEqual({ 1: 1, 2: 1, 3: 1, 4: 1 });

    // drag yop to the second column
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(renderCounts).toEqual({ 1: 2, 2: 1, 3: 1, 4: 1 });

    saveDef.resolve();
    await animationFrame();

    expect(renderCounts).toEqual({ 1: 3, 2: 1, 3: 1, 4: 1 });

    resequenceDef.resolve();
    await animationFrame();

    expect(renderCounts).toEqual({ 1: 4, 2: 1, 3: 1, 4: 1 });

    // drag gnap to the second column
    saveDef = new Deferred();
    resequenceDef = new Deferred();
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(renderCounts).toEqual({ 1: 4, 2: 1, 3: 2, 4: 1 });

    saveDef.resolve();
    await animationFrame();

    expect(renderCounts).toEqual({ 1: 4, 2: 1, 3: 3, 4: 1 });

    resequenceDef.resolve();
    await animationFrame();

    expect(renderCounts).toEqual({ 1: 4, 2: 1, 3: 4, 4: 1 });

    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "get_views",
        "web_read_group",
        "has_group",
        "web_save",
        "web_resequence",
        "web_save",
        "web_resequence",
    ]);
});

test("sample server: _mockWebReadGroup API", async () => {
    Partner._records = [];

    patchWithCleanup(SampleServer.prototype, {
        async _mockWebReadGroup() {
            const result = await super._mockWebReadGroup(...arguments);
            const { "date:month": dateValue } = result.groups[0];
            expect(dateValue[1]).toBe("December 2022");
            return result;
        },
    });

    onRpc("web_read_group", () => ({
        groups: [
            {
                __count: 0,
                __records: [],
                state: false,
                "date:month": ["2022-12-01", "December 2022"],
                __extra_domain: [
                    ["date", ">=", "2022-12-01"],
                    ["date", "<", "2023-01-01"],
                ],
            },
        ],
        length: 1,
    }));

    await mountView({
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="display_name"/>
                    </div>
                </templates>
            </kanban>`,
        groupBy: ["date:month"],
        resModel: "partner",
        type: "kanban",
        noContentHelp: "No content helper",
    });

    expect(".o_kanban_view .o_view_sample_data").toHaveCount(1);
    expect(".o_kanban_group").toHaveCount(1);
    expect(".o_kanban_group .o_column_title").toHaveText("December 2022");
    expect(".o_kanban_group .o_kanban_record").toHaveCount(16);
});

test.tags("desktop");
test(`kanban view: press "hotkey" to execute header button action`, async () => {
    mockService("action", {
        doActionButton(params) {
            const { name } = params;
            expect.step(`execute_action: ${name}`);
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
                <header>
                    <button name="display" type="object" class="display" string="display" display="always" data-hotkey="a"/>
                </header>
                <field name="bar" />
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });

    await press(["alt", "a"]);
    await tick();
    expect.verifySteps(["execute_action: display"]);
});

test.tags("desktop");
test("action button in controlPanel with display='always'", async () => {
    const domain = [["id", "=", 1]];

    mockService("action", {
        async doActionButton(params) {
            const { buttonContext, context, name, resModel, resIds, type } = params;
            expect.step("execute_action");
            // Action's own properties
            expect(name).toBe("display");
            expect(type).toBe("object");

            // The action's execution context
            expect(buttonContext).toEqual({
                active_domain: domain,
                active_ids: [],
                active_model: "partner",
            });

            expect(context).toEqual({
                a: true,
                allowed_company_ids: [1],
                lang: "en",
                tz: "taht",
                uid: 7,
            });
            expect(resModel).toBe("partner");
            expect(resIds).toEqual([]);
        },
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
                <header>
                    <button name="display" type="object" class="display" string="display" display="always"/>
                    <button name="display" type="object" class="display_invisible" string="invisible 1" display="always" invisible="1"/>
                    <button name="display" type="object" class="display_invisible_2" string="invisible context" display="always" invisible="context.get('a')"/>
                    <button name="default-selection" type="object" class="default-selection" string="default-selection"/>
                </header>
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                    </t>
                </templates>
            </kanban>`,
        domain,
        context: {
            a: true,
        },
    });

    const cpButtons = queryAll(".o_control_panel_main_buttons button:visible");
    expect(queryAllTexts(cpButtons)).toEqual(["New", "display"]);
    expect(cpButtons[1]).toHaveClass("display");

    await contains(cpButtons[1]).click();

    expect.verifySteps(["execute_action"]);
});

test.tags("desktop");
test("Keep scrollTop when loading records with load more", async () => {
    Partner._records[0].bar = false;
    Partner._records[1].bar = false;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <div style="height:1000px;"><field name="id"/></div>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["bar"],
        limit: 1,
    });
    const clickKanbanLoadMoreButton = queryFirst(".o_kanban_load_more button");
    clickKanbanLoadMoreButton.scrollIntoView();
    const previousScrollTop = queryOne(".o_content").scrollTop;
    clickKanbanLoadMoreButton.click();
    await animationFrame();
    expect(previousScrollTop).not.toBe(0, { message: "Should not have the scrollTop value at 0" });
    expect(queryOne(".o_content").scrollTop).toBe(previousScrollTop);
});

test("Kanban: no reset of the groupby when a non-empty column is deleted", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="groupby_category" string="Category" context="{'group_by': 'category_ids'}"/>
            </search>`,
    });

    // validate presence of the search arch info
    await toggleSearchBarMenu();
    expect(".o_group_by_menu span.o_menu_item").toHaveCount(1);

    // select the groupby:category_ids filter
    await contains(".o_group_by_menu span.o_menu_item").click();
    // check the initial rendering
    expect(".o_kanban_group").toHaveCount(3, { message: "should have three columns" });

    // check availability of delete action in kanban header's config dropdown
    await toggleKanbanColumnActions(2);
    expect(queryAll(".o_group_delete", { root: getKanbanColumnDropdownMenu(2) })).toHaveCount(1, {
        message: "should be able to delete the column",
    });

    // delete second column (first cancel the confirm request, then confirm)
    let clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Delete");
    await contains(".o_dialog footer .btn-secondary").click();

    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("gold\n(1)");

    clickColumnAction = await toggleKanbanColumnActions(1);
    await clickColumnAction("Delete");
    await contains(".o_dialog footer .btn-primary").click();

    expect(".o_kanban_group").toHaveCount(2, { message: "should now have two columns" });
    expect(queryText(".o_column_title", { root: getKanbanColumn(1) })).toBe("silver\n(1)");
    expect(queryText(".o_column_title", { root: getKanbanColumn(0) })).toBe("None\n(3)");
});

test.tags("desktop");
test("searchbar filters are displayed directly", async () => {
    let def;
    onRpc("web_search_read", () => def);

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="some_filter" string="Some Filter" domain="[['foo', '!=', 'bar']]"/>
            </search>`,
    });

    expect(getFacetTexts()).toEqual([]);

    // toggle a filter, and slow down the web_search_read rpc
    def = new Deferred();
    await toggleSearchBarMenu();
    await toggleMenuItem("Some Filter");
    expect(getFacetTexts()).toEqual(["Some Filter"]);

    def.resolve();
    await animationFrame();
    expect(getFacetTexts()).toEqual(["Some Filter"]);
});

test.tags("desktop");
test(`searchbar in kanban view doesn't take focus after unselected all items`, async () => {
    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    expect(`.o_searchview_input`).toBeFocused({
        message: "The search input should have the focus",
    });

    await contains(`.o_kanban_record`).click({ altKey: true });
    await contains(`.o_kanban_record.o_record_selected`).click();
    expect(`.o_searchview_input`).not.toBeFocused({
        message: "The search input shouldn't have the focus",
    });
});

test.tags("desktop");
test("group by properties and drag and drop", async () => {
    expect.assertions(7);

    Partner._fields.properties = fields.Properties({
        definition_record: "parent_id",
        definition_record_field: "properties_definition",
    });
    Partner._fields.parent_id = fields.Many2one({ relation: "partner" });
    Partner._fields.properties_definition = fields.PropertiesDefinition();

    Partner._records[0].properties_definition = [
        {
            name: "my_char",
            string: "My Char",
            type: "char",
        },
    ];
    Partner._records[1].parent_id = 1;
    Partner._records[2].parent_id = 1;
    Partner._records[3].parent_id = 2;

    onRpc("web_read_group", () => ({
        groups: [
            {
                "properties.my_char": false,
                __extra_domain: [["properties.my_char", "=", false]],
                __count: 2,
            },
            {
                "properties.my_char": "aaa",
                __extra_domain: [["properties.my_char", "=", "aaa"]],
                __count: 1,
                __records: [
                    {
                        id: 2,
                        properties: [
                            {
                                name: "my_char",
                                string: "My Char",
                                type: "char",
                                value: "aaa",
                            },
                        ],
                    },
                ],
            },
            {
                "properties.my_char": "bbb",
                __extra_domain: [["properties.my_char", "=", "bbb"]],
                __count: 1,
                __records: [
                    {
                        id: 3,
                        properties: [
                            {
                                name: "my_char",
                                string: "My Char",
                                type: "char",
                                value: "bbb",
                            },
                        ],
                    },
                ],
            },
        ],
        length: 3,
    }));
    onRpc("web_resequence", () => {
        expect.step("resequence");
        return [];
    });
    onRpc("web_save", ({ args }) => {
        expect.step("web_save");
        const expected = {
            properties: [
                {
                    name: "my_char",
                    string: "My Char",
                    type: "char",
                    value: "bbb",
                },
            ],
        };
        expect(args[1]).toEqual(expected);
    });
    onRpc("get_property_definition", ({ args }) => {
        expect.step("get_property_definition");
        return {
            name: "my_char",
            type: "char",
        };
    });

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="properties"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["properties.my_char"],
    });

    expect.verifySteps(["get_property_definition"]);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(1);

    await contains(".o_kanban_group:nth-child(2) .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(3)"
    );

    expect.verifySteps(["web_save", "resequence"]);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(3) .o_kanban_record").toHaveCount(2);
});

test("kanbans with basic and custom compiler, same arch", async () => {
    // In this test, the exact same arch will be rendered by 2 different kanban renderers:
    // once with the basic one, and once with a custom renderer having a custom compiler. The
    // purpose of the test is to ensure that the template is compiled twice, once by each
    // compiler, even though the arch is the same.
    class MyKanbanCompiler extends KanbanCompiler {
        setup() {
            super.setup();
            this.compilers.push({ selector: "div", fn: this.compileDiv });
        }

        compileDiv(node, params) {
            const compiledNode = this.compileGenericNode(node, params);
            compiledNode.setAttribute("class", "my_kanban_compiler");
            return compiledNode;
        }
    }
    viewRegistry.add("my_kanban", {
        ...kanbanView,
        Compiler: MyKanbanCompiler,
    });
    after(() => viewRegistry.remove("my_kanban"));

    Partner._fields.one2many = fields.One2many({ relation: "partner" });
    Partner._records[0].one2many = [1];
    Partner._views["form"] = `<form><field name="one2many" mode="kanban"/></form>`;
    Partner._views["kanban"] = `
        <kanban js_class="my_kanban">
            <templates>
                <t t-name="card">
                    <div><field name="foo"/></div>
                </t>
            </templates>
        </kanban>`;

    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
    });

    // main kanban, custom view
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_my_kanban_view").toHaveCount(1);
    expect(".my_kanban_compiler").toHaveCount(4);

    // switch to form
    await contains(".o_kanban_record").click();
    await animationFrame();
    expect(".o_form_view").toHaveCount(1);
    expect(".o_form_view .o_field_widget[name=one2many]").toHaveCount(1);

    // x2many kanban, basic renderer
    expect(".o_kanban_record:not(.o_kanban_ghost):not(.o-kanban-button-new)").toHaveCount(1);
    expect(".my_kanban_compiler").toHaveCount(0);
});

test("grouped on field with readonly expression depending on context", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="product_id" readonly="context.get('abc')" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
        context: { abc: true },
    });

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);
});

test.tags("desktop");
test("grouped on field with readonly expression depending on fields", async () => {
    // Fields are not available in the current context as the drag and drop must be enabled globally
    // for the view, it's not a per record thing.
    // So if the readonly expression contains fields, it will resolve to readonly === false and
    // the drag and drop will be enabled.
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo" />
                        <field name="product_id" readonly="foo == 'yop'" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(1);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(3);
});

test.tags("desktop");
test("quick create a column by pressing enter when input is focused", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2);

    await quickCreateKanbanColumn();

    // We don't use the editInput helper as it would trigger a change event automatically.
    // We need to wait for the enter key to trigger the event.
    await press("N");
    await press("e");
    await press("w");
    await press("Enter");
    await animationFrame();

    expect(".o_kanban_group").toHaveCount(3);
});

test("group by numeric field (with aggregator)", async () => {
    onRpc("web_read_group", ({ kwargs }) => {
        expect(kwargs.groupby).toEqual(["int_field"]);
        expect(kwargs.aggregates).toEqual([]); // No progressbar - no aggregate needed
        expect.step("web_read_group");
    });
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban class="o_kanban_test">
                <field name="int_field" />
                <field name="float_field" />
                <templates>
                    <t t-name="card">
                        <div>
                            <field name="foo" />
                        </div>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["int_field"],
    });
    expect.verifySteps(["web_read_group"]);
});

test.tags("desktop");
test("click on empty kanban must shake the NEW button", async () => {
    onRpc("web_read_group", () =>
        // override read_group to return empty groups, as this is
        // the case for several models (e.g. project.task grouped
        // by stage_id)
        ({
            groups: [
                {
                    __extra_domain: [["product_id", "=", 3]],
                    __count: 0,
                    product_id: [3, "xplone"],
                },
                {
                    __extra_domain: [["product_id", "=", 5]],
                    __count: 0,
                    product_id: [5, "xplan"],
                },
            ],
            length: 2,
        })
    );

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o_kanban_group").toHaveCount(2, { message: "there should be 2 columns" });
    expect(".o_kanban_record").toHaveCount(0, { message: "both columns should be empty" });

    await click(".o_kanban_renderer");

    expect("[data-bounce-button]").toHaveClass("o_catch_attention");
});

test.tags("mobile");
test("Should load grouped kanban with folded column", async () => {
    Product._records[1].fold = true;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <progressbar field="foo" colors='{"yop": "success", "blip": "danger"}'/>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_column_progress").toHaveCount(2, { message: "Should have 2 progress bar" });
    expect(".o_kanban_group").toHaveCount(2, { message: "Should have 2 grouped column" });
    expect(".o_kanban_record").toHaveCount(2, { message: "Should have 2 loaded record" });
    expect(".o_kanban_load_more").toHaveCount(1, {
        message: "Should have a folded column with a load more button",
    });
    await contains(".o_kanban_load_more button").click();
    expect(".o_kanban_load_more").toHaveCount(0, { message: "Shouldn't have a load more button" });
    expect(".o_kanban_record").toHaveCount(4, { message: "Should have 4 loaded record" });
});

test("kanban records are middle clickable by default", async () => {
    patchWithCleanup(browser, {
        open: (url) => {
            expect.step(`opened in new window: ${url}`);
        },
    });
    patchWithCleanup(browser.sessionStorage, {
        setItem(key, value) {
            expect.step(`set ${key}-${value}`);
            super.setItem(key, value);
        },
        getItem(key) {
            const res = super.getItem(key);
            expect.step(`get ${key}-${res}`);
            return res;
        },
    });
    Partner._views.kanban = /* xml */ `
        <kanban>
            <templates>
                <t t-name="card">
                    <field name="foo"/>
                </t>
            </templates>
        </kanban>`;
    Partner._views.form = /* xml */ `
        <form>
            <field name="product_id"/>
            <field name="foo"/>
        </form>`;

    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        id: 1,
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
    });

    await contains(".o_kanban_record").click({ ctrlKey: true });
    expect.verifySteps([
        "get menu_id-null",
        "get current_lang-null",
        "get current_state-null",
        "get current_action-null",
        'set current_state-{"actionStack":[{"displayName":"","action":1,"view_type":"kanban"}],"action":1}',
        'set current_action-{"id":1,"res_model":"partner","type":"ir.actions.act_window","views":[[false,"kanban"],[false,"form"]]}',
        "set current_lang-en",
        'get current_action-{"id":1,"res_model":"partner","type":"ir.actions.act_window","views":[[false,"kanban"],[false,"form"]]}',
        'get current_state-{"actionStack":[{"displayName":"","action":1,"view_type":"kanban"}],"action":1}',
        'set current_action-{"id":1,"res_model":"partner","type":"ir.actions.act_window","views":[[false,"kanban"],[false,"form"]]}',
        'set current_state-{"actionStack":[{"displayName":"","action":1,"view_type":"kanban"},{"displayName":"","action":1,"view_type":"form","resId":1}],"resId":1,"action":1}',
        "opened in new window: /odoo/action-1/1",
        'set current_action-{"id":1,"res_model":"partner","type":"ir.actions.act_window","views":[[false,"kanban"],[false,"form"]]}',
        'set current_state-{"actionStack":[{"displayName":"","action":1,"view_type":"kanban"}],"action":1}',
    ]);
});

test("display 'None' for false group, when grouped by char field", async () => {
    Partner._records[0].foo = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["foo"],
    });

    expect(".o_kanban_group:first-child .o_column_title").toHaveText("None\n(1)");
});

test("display '0' for false group, when grouped by int field", async () => {
    Partner._records[0].int_field = 0;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="int_field"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["int_field"],
    });

    expect(".o_kanban_group:first-child .o_column_title").toHaveText("0\n(1)");
});

test("display the field's falsy_value_label for false group, if defined", async () => {
    Partner._fields.foo.falsy_value_label = "I'm the false group";
    Partner._records[0].foo = false;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["foo"],
    });

    expect(".o_kanban_group:first-child .o_column_title").toHaveText("I'm the false group\n(1)");
});

test("selection can be enabled with the 'alt' key", async () => {
    Product._records[1].fold = true;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_selection_box").toHaveCount(0);
    await keyDown("alt");
    await animationFrame();
    expect(".o_kanban_record").toHaveClass("o_record_selection_available");
    expect(".o_kanban_record > .o_record_selection_tooltip").toHaveText("Click to select");
    await contains(".o_kanban_record:nth-of-type(1)").click();
    expect(".o_selection_box").toHaveCount(1);
    await keyUp("alt");
    await animationFrame();
    await contains(".o_kanban_record:nth-of-type(2)").click();
    expect(".o_selection_box span > b").toHaveText("2", {
        message: "selection counter has the right number of selected items",
    });
    expect(".o_record_selected").toHaveCount(2);
});

test.tags("desktop");
test("selection is reset when dragging is effective", async () => {
    mockTouch(true);
    Product._records[1].fold = true;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_selection_box").toHaveCount(0);
    const { moveTo } = await drag(".o_kanban_record:nth-of-type(1)");
    await advanceTime(600);
    expect(".o_selection_box").toHaveCount(1);
    await moveTo(".o_kanban_group:nth-of-type(2)");
    await runAllTimers();
    expect(".o_selection_box").toHaveCount(0);
});

test("selection can be enabled by long touch", async () => {
    mockTouch(true);
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
    });
    expect(".o_selection_box").toHaveCount(0);
    await drag(".o_kanban_record:nth-of-type(2)");
    await advanceTime(TOUCH_SELECTION_THRESHOLD);
    expect(".o_selection_box").toHaveCount(1);
});

test("selection can be enabled by long touch with drag & drop enabled", async () => {
    mockTouch(true);
    Product._records[1].fold = true;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
        groupBy: ["product_id"],
    });
    expect(".o_selection_box").toHaveCount(0);
    const { drop } = await drag(".o_kanban_record:nth-of-type(1)");
    await advanceTime(TOUCH_SELECTION_THRESHOLD);
    expect(".o_selection_box").toHaveCount(0, {
        message: "touch delay is longer when drag & drop is enabled",
    });
    await drop();
    const { drop: secondDrop } = await drag(".o_kanban_record:nth-of-type(1)");
    await advanceTime(600);
    expect(".o_selection_box").toHaveCount(1);
    await secondDrop();
    await contains(".o_kanban_record:nth-of-type(2)").click();
    expect(".o_selection_box span > b").toHaveText("2", {
        message: "selection counter has the right number of selected items",
    });
    expect(".o_record_selected").toHaveCount(2);
    await contains(".o_kanban_record:nth-of-type(1)").click();
    expect(".o_record_selected").toHaveCount(1);
});

test.tags("desktop");
test("selection can be enabled by pressing 'space' key", async () => {
    Product._records[1].fold = true;
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
    });
    expect(".o_selection_box").toHaveCount(0);
    await press("ArrowDown");
    await press("Space");
    await animationFrame();
    expect(".o_selection_box").toHaveCount(1);
    await press("ArrowDown");
    await press("Space");
    await animationFrame();
    expect(".o_record_selected").toHaveCount(2);
    await press("ArrowDown");
    await press("ArrowDown");
    await keyDown("Shift");
    await press("Space");
    await animationFrame();
    expect(".o_record_selected").toHaveCount(4);
});

test.tags("desktop");
test("selection can be enabled by pressing 'shift + space' key", async () => {
    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
                <kanban>
                    <templates>
                        <t t-name="card">
                            <field name="foo"/>
                        </t>
                    </templates>
                </kanban>`,
    });
    expect(".o_selection_box").toHaveCount(0);
    await press("ArrowDown");
    await keyDown("Shift");
    await press("Space");
    await animationFrame();
    expect(".o_record_selected").toHaveCount(1);
    await keyUp("Shift");
    await press("ArrowDown");
    await press("ArrowDown");
    await keyDown("Shift");
    await press("Space");
    await animationFrame();
    expect(".o_record_selected").toHaveCount(3);
});

test.tags("desktop");
test("drag and drop records and quickly open a record", async () => {
    Partner._views.kanban = /* xml */ `
        <kanban>
            <templates>
                <t t-name="card">
                    <div><field name="foo"/></div>
                </t>
            </templates>
        </kanban>`;
    Partner._views.form = /* xml */ `
        <form>
            <field name="foo"/>
        </form>`;

    const defs = [new Deferred(), new Deferred()];
    let saveCount = 0;
    onRpc("web_save", () => {
        expect.step("web_save");
        return defs[saveCount++];
    });

    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "form"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(2);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(2);

    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );
    await contains(".o_kanban_group:first-child .o_kanban_record").dragAndDrop(
        ".o_kanban_group:nth-child(2)"
    );
    await contains(".o_kanban_record:eq(0)").click();
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(4);
    expect.verifySteps(["web_save"]);

    defs[0].resolve();
    await animationFrame();
    // because of the mutex in the model, the second web_save is done only once the first one
    // returned, but that rpc can't be done if the component has already been destroyed.
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_group:first-child .o_kanban_record").toHaveCount(0);
    expect(".o_kanban_group:nth-child(2) .o_kanban_record").toHaveCount(4);
    expect.verifySteps(["web_save"]);

    defs[1].resolve();
    await animationFrame();
    expect(".o_form_view").toHaveCount(1);
});

test.tags("desktop");
test("groups will be scrolled to on unfold if outside of viewport", async () => {
    for (let i = 0; i < 12; i++) {
        Product._records.push({ id: 8 + i, name: `column ${i}` });
        Partner._records.push({ id: 20 + i, foo: "dumb entry", product_id: 8 + i });
    }
    Product._records[2].fold = true;
    Product._records[8].fold = true;
    Product._records[9].fold = true;

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
    });
    disableAnimations();
    expect(".o_content").toHaveProperty("scrollLeft", 0);
    await contains(".o_column_folded:eq(0)").click();
    await animationFrame();
    // Group completely inside the viewport after unfold, no scroll
    expect(".o_content").toHaveProperty("scrollLeft", 0);
    await contains(".o_content").scroll({ left: 1500 });
    await contains(".o_column_folded:eq(0)").click();
    // Group is followed by a folded group which is outside the viewport
    // after unfold, scroll to that group
    expect(".o_content").toHaveProperty("scrollLeft", 1844);
    let { x, width } = queryRect(".o_column_folded:eq(0)");
    // TODO JUM: change digits option
    expect(x + width).toBeCloseTo(window.innerWidth - 1, {
        digits: 0,
        message:
            "the next group (which is folded) should stick to the right of the screen after the scroll",
    });
    expect(".o_column_folded:eq(0)").toHaveText("column 7 (1)", { inline: true });
    await contains('.o_kanban_group:contains("column 7 (1)")').click();
    expect(".o_content").toHaveProperty("scrollLeft", 2154);
    ({ x, width } = queryRect('.o_kanban_group:contains("column 7 (1)")'));
    // TODO JUM: change digits option
    expect(x + width).toBeCloseTo(window.innerWidth, {
        digits: 0,
        message:
            "this group was not followed by a folded group so it will be the one to stick to the right of the screen after the scroll",
    });
    // scroll to the end
    await contains(".o_content").scroll({ left: 5000 });
    expect(".o_content").toHaveProperty("scrollLeft", 3302);
    await contains(".o_kanban_group:last").click();
    expect(".o_content").toHaveProperty("scrollLeft", 3562);
    ({ x, width } = queryRect('.o_kanban_group:contains("column 11 (1)")'));
    // TODO JUM: change digits option
    expect(x + width).toBeCloseTo(window.innerWidth, {
        digits: 0,
        message: "same as above",
    });
});

test("hide pager in the kanban view with sample data", async () => {
    Partner._records = [];

    await mountView({
        arch: `
            <kanban sample="1">
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                        <field name="int_field"/>
                    </div>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
        noContentHelp: "No content helper",
    });

    expect(".o_content").toHaveClass("o_view_sample_data");
    expect(".o_cp_pager").not.toHaveCount();
});

test.tags("desktop");
test("kanban views make their control panel available directly", async () => {
    const def = new Deferred();
    onRpc("web_search_read", () => def);
    await mountView({
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_view .o_control_panel .o_searchview").toHaveCount(1);
    expect(".o_kanban_view .o_kanban_renderer").toHaveCount(0);

    def.resolve();
    await animationFrame();
    expect(".o_kanban_view .o_kanban_renderer").toHaveCount(1);
    expect(".o_kanban_view .o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
});

test.tags("desktop");
test("interact with search view while kanban is loading", async () => {
    const defs = [new Deferred()];
    onRpc("web_search_read", () => defs.pop());
    await mountView({
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter name="group_by_foo" domain="[]" string="GroupBy Foo" context="{ 'group_by': 'foo' }"/>
            </search>`,
        resModel: "partner",
        type: "kanban",
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_view .o_control_panel .o_searchview").toHaveCount(1);
    expect(".o_kanban_view .o_kanban_renderer").toHaveCount(0);

    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Foo");
    expect(".o_kanban_view .o_kanban_renderer").toHaveCount(1);
    expect(".o_kanban_view .o_kanban_group").toHaveCount(3);
    expect(".o_kanban_view .o_kanban_record").toHaveCount(4);
});

test("click on New while kanban is loading", async () => {
    onRpc("web_search_read", () => new Deferred());
    await mountView({
        arch: `
            <kanban>
                <templates>
                    <div t-name="card">
                        <field name="foo"/>
                    </div>
                </templates>
            </kanban>`,
        resModel: "partner",
        type: "kanban",
        createRecord: () => expect.step("create record"),
    });

    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_view .o_control_panel").toHaveCount(1);
    expect(".o_kanban_view .o_kanban_renderer").toHaveCount(0);

    await createKanbanRecord();
    expect.verifySteps(["create record"]);
});

test(`kanban with custom cog action that has a confirmation target="new" action`, async () => {
    const contextualAction = {
        id: 80,
        name: "Sort of confirmation dialog",
        res_model: "partner",
        context: "{}",
        views: [[false, "form"]],
        type: "ir.actions.act_window",
        target: "new",
    };
    Partner._toolbar = {
        action: [contextualAction],
        print: [],
    };
    Partner._views = {
        kanban: `
            <kanban>
                <t t-name="card">
                    <field name="foo"/>
                </t>
            </kanban>`,
        search: `<search/>`,
        form: `
            <form>
                Are you sure blablabla
                <footer>
                    <button name="my_action" type="action" string="Do it"/>
                </footer>
            </form>`,
    };
    defineActions([
        {
            id: 1,
            name: "Partner",
            res_model: "partner",
            views: [[false, "kanban"]],
        },
        {
            id: 2,
            name: "Partner",
            res_model: "partner",
            views: [[false, "form"]],
            res_id: 1,
            xml_id: "my_action",
        },
        contextualAction,
    ]);

    stepAllNetworkCalls();
    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(".o_kanban_view").toHaveCount(1);

    await keyDown("alt");
    await contains(".o_kanban_record:nth-of-type(1)").click();
    expect(".o_selection_box").toHaveCount(1);
    await contains(`.o_cp_action_menus button:has(.fa-cog)`).click();
    await contains(`.o-dropdown-item:contains(Sort of confirmation dialog)`).click();
    expect(".o_dialog").toHaveCount(1);

    await contains(".o_dialog footer button:contains(Do it)").click();
    expect(".o_dialog").toHaveCount(0);
    expect(".o_form_view").toHaveCount(1);

    // should not reload the list view when confirming with Do it
    expect.verifySteps([
        "/web/webclient/translations",
        "/web/webclient/load_menus",
        "/web/action/load",
        "get_views",
        "web_search_read",
        "has_group",
        "/web/action/load",
        "get_views",
        "onchange",
        "web_save",
        "/web/action/load",
        "get_views",
        "web_read",
    ]);
});

test(`cache web_read_group (no change)`, async () => {
    let def;
    onRpc("web_read_group", () => def);

    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban default_group_by="bar">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_header`)).toEqual(["No\n(1)", "Yes\n(3)"]);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_header`)).toEqual(["No\n(1)", "Yes\n(3)"]);

    // simulate the return of web_read_group => nothing should have changed
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_header`)).toEqual(["No\n(1)", "Yes\n(3)"]);
});

test(`cache web_read_group (change)`, async () => {
    let def;
    onRpc("web_read_group", () => def);

    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban default_group_by="int_field">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(4);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual([
        "-4\n(1)",
        "9\n(1)",
        "10\n(1)",
        "17\n(1)",
    ]);

    // simulate the create of new records by someone else
    MockServer.env.partner.create([{ int_field: 44 }, { int_field: -4 }]);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(4);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual([
        "-4\n(1)",
        "9\n(1)",
        "10\n(1)",
        "17\n(1)",
    ]);

    // simulate the return of web_read_group => the data should have been updated
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(5);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual([
        "-4\n(2)",
        "9\n(1)",
        "10\n(1)",
        "17\n(1)",
        "44\n(1)",
    ]);
});

test(`cache web_read_group (no data, no change)`, async () => {
    let def;
    onRpc("web_read_group", () => def);

    Partner._records = [];
    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);

    // simulate the return of web_read_group => the sample data should still be displayed
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);
});

test(`cache web_read_group (no data, change)`, async () => {
    let def;
    onRpc("web_read_group", () => def);

    Partner._records = [];
    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban sample="1" default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);

    // simulate the create of new records by someone else
    MockServer.env.partner.create([{ product_id: 3 }, { product_id: 5 }]);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);

    // simulate the return of web_read_group => the data should have been updated
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual(["hello\n(1)", "xmo\n(1)"]);
});

test(`cache web_read_group (group_expand: groups, then no group)`, async () => {
    // this test simulates that we are on a grouped kanban, with the group_expand feature, i.e.
    // empty groups are shown, and when we first go to the view, there are groups, but empty, so
    // sample data is displayed. Then, when we come back, we retrieve the data from the cache, but
    // the rpc returns no group, so the view must be properly updated.
    let def;
    let withGroups = true;
    onRpc("web_read_group", async () => {
        if (withGroups) {
            return {
                groups: [
                    {
                        __extra_domain: [["product_id", "=", 3]],
                        __records: [],
                        __count: 0,
                        product_id: [3, "xphone"],
                    },
                ],
                length: 1,
            };
        } else {
            await def;
            return { groups: [], length: 0 };
        }
    });

    Partner._records = [];
    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban sample="1" default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(1);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // simulate the removal of the (only) empty group
    withGroups = false;

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(1);

    // simulate the return of web_read_group => the data should have been updated
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view`).toHaveCount(1);
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(0);
    expect(`.o_kanban_view .o_column_quick_create`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group_nocontent`).toHaveCount(1);
});

test(`cache web_read_group (group_expand: groups, then more groups)`, async () => {
    // this test simulates that we are on a grouped kanban, with the group_expand feature, i.e.
    // empty groups are shown, and when we first go to the view, there are groups, but empty, so
    // sample data is displayed. Then, when we come back, we retrieve the data from the cache, but
    // the rpc returns more (or less, but still some) groups. Theoretically, we should remain in
    // sample mode and display the updated groups. We do correctly display the groups, but the
    // sample mode is left, because it is a quite complex usecase to deal with, and we don't think
    // it would be worth the complexity. This test simply encodes the current behavior, that we may
    // change in the future if we want to.
    let def;
    const groups = [
        {
            __extra_domain: [["product_id", "=", 3]],
            __records: [],
            __count: 0,
            product_id: [3, "xphone"],
        },
    ];
    onRpc("web_read_group", async () => {
        await def;
        return {
            groups,
            length: groups.length,
        };
    });

    Partner._records = [];
    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban sample="1" default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(1);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // simulate the addition of another empty group
    groups.push({
        __extra_domain: [["product_id", "=", 5]],
        __records: [],
        __count: 0,
        product_id: [5, "xpad"],
    });

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(1);

    // simulate the return of web_read_group => the data should have been updated
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view .o_view_sample_data`).toHaveCount(0); // would be ok to remain in sample
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(2);
});

test(`cache web_read_group: less groups than in cache`, async () => {
    // this test simulates that we are on a grouped kanban and the rpc returns less groups than we
    // got from the cache. Those missing groups should be properly removed from the UI on update.
    let def;
    onRpc("web_read_group", () => def);

    Partner._views = {
        "list,false": `<list><field name="foo"/></list>`,
        "kanban,false": `
            <kanban sample="1" default_group_by="product_id">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
        {
            id: 2,
            name: "Another action",
            res_model: "partner",
            views: [[false, "list"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual(["hello\n(2)", "xmo\n(2)"]);

    // execute another action to remove the kanban from the DOM
    await getService("action").doAction(2);
    expect(`.o_list_view`).toHaveCount(1);

    // simulate a move of records from first group to the second one
    MockServer.env.partner.write([1, 3], { product_id: 5 });

    // execute again action 1, but web_read_group is delayed
    def = new Deferred();
    await getService("action").doAction(1);
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(2);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual(["hello\n(2)", "xmo\n(2)"]);

    // simulate the return of web_read_group => the data should have been updated
    def.resolve();
    await animationFrame();
    expect(`.o_kanban_view .o_kanban_group`).toHaveCount(1);
    expect(queryAllTexts(`.o_kanban_group .o_kanban_header`)).toEqual(["xmo\n(4)"]);
});

test.tags("desktop");
test("Cache: folded is now unfolded", async () => {
    Product._records[1].fold = true;

    Partner._views = {
        "kanban,false": `
            <kanban default_group_by="product_id">
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(queryText(getKanbanColumn(1))).toBe("xmo\n(2)");

    MockServer.env["product"].write(5, { fold: false });
    await getService("action").doAction(1);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);
});

test.tags("desktop");
test("Cache: unfolded is now folded", async () => {
    Product._records[1].fold = false;

    Partner._views = {
        "kanban,false": `
            <kanban default_group_by="product_id">
                <templates>
                    <div t-name="card">
                        <field name="id"/>
                    </div>
                </templates>
            </kanban>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [[false, "kanban"]],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);

    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(1) })).toHaveCount(2);

    MockServer.env["product"].write(5, { fold: true });
    await getService("action").doAction(1);
    expect(queryAll(".o_kanban_record", { root: getKanbanColumn(0) })).toHaveCount(2);
    expect(getKanbanColumn(1)).toHaveClass("o_column_folded");
    expect(queryText(getKanbanColumn(1))).toBe("xmo\n(2)");
});

test.tags("desktop");
test("Cache: kanban view progressbar, filter, open a record, edit, come back", async () => {
    // This test encodes a very specify scenario involving a kanban with progressbar, where the
    // filter was lost when coming back due to the cache callback, which removed the groups
    // information.
    Product._records[1].fold = false;

    let def;
    onRpc("web_read_group", () => def);

    Partner._views = {
        "kanban,false": `
            <kanban default_group_by="product_id" on_create="quick_create" quick_create_view="some_view_ref">
                <progressbar field="foo" colors='{"yop": "success", "gnap": "warning", "blip": "danger"}'/>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        "form,false": `<form><field name="product_id" widget="statusbar" options="{'clickable': true}"/></form>`,
        "search,false": `<search/>`,
    };

    defineActions([
        {
            id: 1,
            name: "Partners Action",
            res_model: "partner",
            views: [
                [false, "kanban"],
                [false, "form"],
            ],
            search_view_id: [false, "search"],
        },
    ]);

    await mountWithCleanup(WebClient);
    await getService("action").doAction(1);
    expect(".o_kanban_group").toHaveCount(2);
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(2);

    // Filter the first column with the progressbar
    await contains(".o_column_progress .progress-bar", { root: getKanbanColumn(0) }).click();
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(1);

    // Open a record, then go back, s.t. we populate the cache with the current params of the kanban
    await contains(".o_kanban_group:eq(1) .o_kanban_record").click();
    expect(".o_form_view").toHaveCount(1);
    await contains(".o_back_button").click();
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(1);

    // Open again and make a change which will have an impact on the kanban, then go back
    await contains(".o_kanban_group:eq(1) .o_kanban_record").click();
    expect(".o_form_view").toHaveCount(1);
    await contains(".o_field_widget[name=product_id] button[data-value='3']").click();
    // Slow down the rpc s.t. we first use data from the cache, and then we update
    def = new Deferred();
    await contains(".o_back_button").click();
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(1);

    // Resolve the promise
    def.resolve();
    await animationFrame();
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(1);

    // Open a last time and come back => the filter should still be applied correctly
    await contains(".o_kanban_group:eq(1) .o_kanban_record").click();
    await contains(".o_back_button").click();
    expect(".o_kanban_group:eq(0) .o_kanban_record").toHaveCount(1);
});

test.tags("desktop");
test("scroll position is restored when coming back to kanban view", async () => {
    Partner._views = {
        kanban: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        list: `<list><field name="foo"/></list>`,
        search: `<search />`,
    };

    for (let i = 1; i < 10; i++) {
        Product._records.push({ id: 100 + i, name: `Product ${i}` });
        for (let j = 1; j < 20; j++) {
            Partner._records.push({
                id: 100 * i + j,
                product_id: 100 + i,
                foo: `Record ${i}/${j}`,
            });
        }
    }

    let def;
    onRpc("web_read_group", () => def);
    await resize({ width: 800, height: 300 });
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "list"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_view").toHaveCount(1);
    // simulate scrolls in the kanban view
    queryOne(".o_content").scrollTop = 100;
    queryOne(".o_content").scrollLeft = 400;

    await getService("action").switchView("list");
    expect(".o_list_view").toHaveCount(1);

    // the kanban is "lazy", so it displays the control panel directly, and the renderer later with
    // the data => simulate this and check that the scroll position is correctly restored
    def = new Deferred();
    await getService("action").switchView("kanban");
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_renderer").toHaveCount(0);
    def.resolve();
    await animationFrame();
    expect(".o_kanban_renderer").toHaveCount(1);
    expect(".o_content").toHaveProperty("scrollTop", 100);
    expect(".o_content").toHaveProperty("scrollLeft", 400);
});

test.tags("mobile");
test("scroll position is restored when coming back to kanban view (mobile)", async () => {
    Partner._views = {
        kanban: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        list: `<list><field name="foo"/></list>`,
        search: `<search />`,
    };

    for (let i = 1; i < 20; i++) {
        Partner._records.push({
            id: 100 + i,
            foo: `Record ${i}`,
        });
    }

    let def;
    onRpc("web_search_read", () => def);
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "list"],
        ],
    });

    expect(".o_kanban_view").toHaveCount(1);
    // simulate a scroll in the kanban view
    queryOne(".o_kanban_view").scrollTop = 100;

    await getService("action").switchView("list");
    expect(".o_list_view").toHaveCount(1);

    // the kanban is "lazy", so it displays the control panel directly, and the renderer later with
    // the data => simulate this and check that the scroll position is correctly restored
    def = new Deferred();
    await getService("action").switchView("kanban");
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_renderer").toHaveCount(0);
    def.resolve();
    await animationFrame();
    expect(".o_kanban_renderer").toHaveCount(1);
    expect(".o_kanban_view").toHaveProperty("scrollTop", 100);
});

test.tags("mobile");
test("scroll position is restored when coming back to kanban view (grouped, mobile)", async () => {
    Partner._views = {
        kanban: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        list: `<list><field name="foo"/></list>`,
        search: `<search />`,
    };

    Partner._records = [];
    for (let i = 1; i < 5; i++) {
        Product._records.push({ id: 100 + i, name: `Product ${i}` });
        for (let j = 1; j < 20; j++) {
            Partner._records.push({
                id: 100 * i + j,
                product_id: 100 + i,
                foo: `Record ${i}/${j}`,
            });
        }
    }

    let def;
    onRpc("web_read_group", () => def);
    await resize({ width: 375, height: 667 }); // iphone se
    await mountWithCleanup(WebClient);
    await getService("action").doAction({
        res_model: "partner",
        type: "ir.actions.act_window",
        views: [
            [false, "kanban"],
            [false, "list"],
        ],
        context: {
            group_by: ["product_id"],
        },
    });

    expect(".o_kanban_view").toHaveCount(1);
    // simulate scrolls in the kanban view
    queryOne(".o_kanban_renderer").scrollLeft = 656; // scroll to the third column
    queryAll(".o_kanban_group")[2].scrollTop = 200;

    await getService("action").switchView("list");
    expect(".o_list_view").toHaveCount(1);

    // the kanban is "lazy", so it displays the control panel directly, and the renderer later with
    // the data => simulate this and check that the scroll position is correctly restored
    def = new Deferred();
    await getService("action").switchView("kanban");
    expect(".o_kanban_view").toHaveCount(1);
    expect(".o_kanban_renderer").toHaveCount(0);
    def.resolve();
    await animationFrame();
    expect(".o_kanban_renderer").toHaveCount(1);
    expect(".o_kanban_group:eq(2)").toHaveProperty("scrollTop", 200);
    expect(".o_kanban_renderer").toHaveProperty("scrollLeft", 656);
});

test.tags("desktop");
test("limit is reset when restoring a view after ungrouping", async () => {
    Partner._views["kanban"] = `
        <kanban sample="1">
            <templates>
                <t t-name="card">
                    <field name="foo"/>
                </t>
            </templates>
        </kanban>`;
    Partner._views["list"] = '<list><field name="foo"/></list>';
    Partner._views.search = `
        <search>
            <group>
                <filter name="foo" string="Foo" context="{'group_by': 'foo'}"/>
            </group>
        </search>
    `;

    onRpc("partner", "web_search_read", ({ kwargs }) => {
        const { domain, limit } = kwargs;
        if (!domain.length) {
            expect.step(`limit=${limit}`);
        }
    });

    patchWithCleanup(user, {
        hasGroup: () => true,
    });

    await mountWithCleanup(WebClient);

    await getService("action").doAction({
        type: "ir.actions.act_window",
        id: 450,
        xml_id: "action_450",
        name: "Partners",
        res_model: "partner",
        views: [
            [false, "kanban"],
            [false, "list"],
            [false, "form"],
        ],
        context: { search_default_foo: true },
    });

    await switchView("list");
    await removeFacet("Foo");
    expect.verifySteps(["limit=80"]);
    await switchView("kanban");
    expect.verifySteps(["limit=40"]);
});

test(`groupby use odoomark`, async () => {
    patchWithCleanup(Partner.prototype, {
        _getFormattedDisplayName(record) {
            return `**${record.name}** \`tag\``;
        },
    });

    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                        <field name="product_id"/>
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(`.o_kanban_group`).toHaveCount(2);
    expect(`.o_kanban_group b`).toHaveCount(2);
    expect(`.o_kanban_group span.o_badge`).toHaveCount(2);
});

test.tags("desktop");
test("kanban: fields with data-tooltip attribute", async () => {
    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban sample="1">
                <templates>
                    <t t-name="card">
                        <field name="foo" data-tooltip="pipu" />
                    </t>
                </templates>
            </kanban>`,
        groupBy: ["product_id"],
    });

    expect(".o-tooltip").toHaveCount(0);
    await hover("article:contains(gnap) span");
    await advanceTime(500);
    expect(".o-tooltip").toHaveCount(1);
});

test.tags("desktop");
test("add o-navigable to buttons with dropdown-item class and view buttons", async () => {
    Partner._records.splice(1, 3); // keep one record only

    await mountView({
        type: "kanban",
        resModel: "partner",
        arch: `
            <kanban>
                <templates>
                    <t t-name="menu">
                        <a role="menuitem" class="dropdown-item">Item</a>
                        <a role="menuitem" type="set_cover" class="dropdown-item">Item</a>
                        <a role="menuitem" type="object" class="dropdown-item">Item</a>
                    </t>
                    <t t-name="card">
                        <div/>
                    </t>
                </templates>
            </kanban>`,
    });

    expect(".o-dropdown--menu").toHaveCount(0);
    await toggleKanbanRecordDropdown();
    expect(".o-dropdown--menu .dropdown-item.o-navigable").toHaveCount(3);
    expect(".o-dropdown--menu .dropdown-item.o-navigable.focus").toHaveCount(0);

    // Check that navigation is working
    await hover(".o-dropdown--menu .dropdown-item.o-navigable");
    expect(".o-dropdown--menu .dropdown-item.o-navigable.focus").toHaveCount(1);

    await press("arrowdown");
    expect(".o-dropdown--menu .dropdown-item.o-navigable:nth-child(2)").toHaveClass("focus");

    await press("arrowdown");
    expect(".o-dropdown--menu .dropdown-item.o-navigable:nth-child(3)").toHaveClass("focus");
});

test.tags("desktop");
test(`[Offline] disable unavailable records when offline`, async () => {
    const setOffline = mockOffline();
    mockService("offline", {
        isAvailableOffline(actionId, viewType, resId) {
            if (actionId === 234 && viewType === "form") {
                return [2, 3].includes(resId);
            }
            return actionId === 123;
        },
    });
    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        config: { actionId: 234 },
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    await setOffline(true);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);
    expect(".o_kanban_record:not(.o_kanban_ghost):eq(0)").toHaveClass("o_disabled_offline");
    expect(".o_kanban_record:not(.o_kanban_ghost):eq(1)").not.toHaveClass("o_disabled_offline");
    expect(".o_kanban_record:not(.o_kanban_ghost):eq(2)").not.toHaveClass("o_disabled_offline");
    expect(".o_kanban_record:not(.o_kanban_ghost):eq(3)").toHaveClass("o_disabled_offline");
});

test.tags("desktop");
test(`[Offline] use offline searchbar`, async () => {
    expect.errors(2);
    const setOffline = mockOffline();
    await mountView({
        resModel: "partner",
        type: "kanban",
        arch: `
            <kanban>
                <templates>
                    <t t-name="card">
                        <field name="foo"/>
                    </t>
                </templates>
            </kanban>`,
        searchViewArch: `
            <search>
                <filter string="Filter Blip" name="blip" domain="[['foo', '=', 'blip']]"/>
                <filter string="GroupBy Blip" name="groupby_blip" context="{'group_by': 'foo'}"/>
                <filter string="Empty Filter" name="empty" domain="[['foo', '=', 'no record']]"/>
            </search>`,
        config: { actionId: 234 },
    });

    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    // Visit filters to put feed the cache
    await toggleSearchBarMenu();
    await toggleMenuItem("GroupBy Blip");
    expect(".o_kanban_group").toHaveCount(3);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    await toggleMenuItem("Filter Blip");
    expect(".o_kanban_group").toHaveCount(1);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);

    await toggleMenuItem("Empty Filter"); // should not be available offline as no results
    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(0);

    await removeFacet("Empty Filter");
    expect(".o_kanban_group").toHaveCount(1);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(2);

    // Switch offline and visit available filters
    await setOffline(true);
    expect(".o_offline_search_bar").toHaveCount(1);
    expect(".o_offline_search_bar .o_searchview_facet").toHaveCount(3); // groupby, filter and close
    expect(queryAllTexts(".o_offline_search_bar .o_facet_values")).toEqual([
        "GroupBy Blip",
        "Filter Blip",
    ]);

    await contains(".o_offline_search_bar .o_searchview_facet .oi-close").click(); // remove search
    expect(".o_offline_search_bar .o_searchview_facet").toHaveCount(0);
    expect(".o_kanban_group").toHaveCount(0);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    await toggleSearchBarMenu();
    expect(".o_search_bar_menu_offline .o-dropdown-item").toHaveCount(2);
    expect(queryAllTexts(".o_search_bar_menu_offline .o-dropdown-item")).toEqual([
        "GroupBy Blip\nFilter Blip",
        "GroupBy Blip",
    ]);

    await contains(".o_search_bar_menu_offline .o-dropdown-item:eq(1)").click();
    expect(".o_offline_search_bar .o_searchview_facet").toHaveCount(2); // groupby and close
    expect(queryAllTexts(".o_offline_search_bar .o_facet_values")).toEqual(["GroupBy Blip"]);
    expect(".o_kanban_group").toHaveCount(3);
    expect(".o_kanban_record:not(.o_kanban_ghost)").toHaveCount(4);

    expect.verifyErrors([
        "/web/dataset/call_kw/partner/web_search_read",
        "/web/dataset/call_kw/partner/web_read_group",
    ]);
});
