# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import api, fields, models
from odoo.tools.misc import clean_context


class StockWarnInsufficientQty(models.AbstractModel):
    _name = 'stock.warn.insufficient.qty'
    _description = 'Warn Insufficient Quantity'

    product_id = fields.Many2one('product.product', 'Product', required=True)
    location_id = fields.Many2one('stock.location', 'Location', domain="[('usage', '=', 'internal')]", required=True)
    quant_ids = fields.Many2many('stock.quant', compute='_compute_quant_ids')
    quantity = fields.Float(string="Quantity", required=True)
    product_uom_name = fields.Char("Unit", required=True)

    def _get_reference_document_company_id(self):
        raise NotImplementedError()

    @api.depends('product_id')
    def _compute_quant_ids(self):
        for quantity in self:
            company = quantity._get_reference_document_company_id()
            quantity.quant_ids = self.env['stock.quant'].search([
                *self.env['stock.quant']._check_company_domain(company),
                ('product_id', '=', quantity.product_id.id),
                ('location_id.usage', '=', 'internal'),
            ])

    def action_done(self):
        raise NotImplementedError()


class StockWarnInsufficientQtyScrap(models.TransientModel):
    _name = 'stock.warn.insufficient.qty.scrap'
    _inherit = ['stock.warn.insufficient.qty']
    _description = 'Warn Insufficient Scrap Quantity'

    scrap_move_id = fields.Many2one('stock.move', 'Scrap')

    def _get_reference_document_company_id(self):
        return self.scrap_move_id.company_id

    def action_done(self):
        return self.with_context(clean_context(self.env.context)).scrap_move_id._action_scrap()

    def action_cancel(self):
        return self.scrap_move_id.sudo().unlink()
