# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import api, fields, models, _
from odoo.exceptions import UserError
from odoo.tools import float_is_zero


class ChangeProductionQty(models.TransientModel):
    _name = 'change.production.qty'
    _description = 'Change Production Qty'

    mo_id = fields.Many2one('mrp.production', 'Manufacturing Order',
        required=True, ondelete='cascade')
    product_qty = fields.Float(
        'Quantity To Produce',
        digits='Product Unit', required=True)

    @api.model
    def default_get(self, fields):
        res = super(ChangeProductionQty, self).default_get(fields)
        if 'mo_id' in fields and not res.get('mo_id') and self.env.context.get('active_model') == 'mrp.production' and self.env.context.get('active_id'):
            res['mo_id'] = self.env.context['active_id']
        if 'product_qty' in fields and not res.get('product_qty') and res.get('mo_id'):
            res['product_qty'] = self.env['mrp.production'].browse(res['mo_id']).product_qty
        return res

    @api.model
    def _update_finished_moves(self, production, new_qty, old_qty):
        """ Update finished product and its byproducts. This method only update
        the finished moves not done or cancel and just increase or decrease
        their quantity according the unit_ratio. It does not use the BoM, BoM
        modification during production would not be taken into consideration.
        """
        modification = {}
        push_moves = self.env['stock.move']
        for move in production.move_finished_ids:
            if move.state in ('done', 'cancel'):
                continue
            qty = (new_qty - old_qty) * move.unit_factor
            modification[move] = (move.product_uom_qty + qty, move.product_uom_qty)
            if self._need_quantity_propagation(move, qty):
                push_moves |= move.copy({'product_uom_qty': qty})
            else:
                move.write({'product_uom_qty': move.product_uom_qty + qty})

        if push_moves:
            push_moves._action_confirm()
        production.move_finished_ids._action_assign()

        return modification

    @api.model
    def _need_quantity_propagation(self, move, qty):
        return move.move_dest_ids and not move.uom_id.is_zero(qty)

    def change_prod_qty(self):
        precision = self.env['decimal.precision'].precision_get('Product Unit')
        for wizard in self:
            production = wizard.mo_id
            old_production_qty = production.product_qty
            new_production_qty = wizard.product_qty

            factor = new_production_qty / old_production_qty
            update_info = production._update_raw_moves(factor)
            documents = {}
            for move, old_qty, new_qty in update_info:
                iterate_key = production._get_document_iterate_key(move)
                if iterate_key:
                    document = self.env['stock.picking']._log_activity_get_documents({move: (new_qty, old_qty)}, iterate_key, 'UP')
                    for key, value in document.items():
                        if documents.get(key):
                            documents[key] += [value]
                        else:
                            documents[key] = [value]
            production._log_manufacture_exception(documents)
            self._update_finished_moves(production, new_production_qty, old_production_qty)
            production.write({'product_qty': new_production_qty})
            if not production.uom_id.is_zero(production.qty_producing) and not production.workorder_ids:
                production.qty_producing = new_production_qty
                production._set_qty_producing()

            for wo in production.workorder_ids:
                quantity = wo.qty_production - wo.qty_produced
                if production.product_id.tracking == 'serial':
                    quantity = 1.0 if not float_is_zero(quantity, precision_digits=precision) else 0.0
                else:
                    quantity = quantity if (quantity > 0 and not float_is_zero(quantity, precision_digits=precision)) else 0
                wo._update_qty_producing(quantity)
                wo.duration_expected = wo._get_duration_expected(ratio=new_production_qty / old_production_qty)
                if wo.qty_produced < wo.qty_production and wo.state == 'done':
                    wo.state = 'progress'
                if wo.qty_produced == wo.qty_production and wo.state == 'progress':
                    wo.state = 'done'

            # replan production based on new workorder durations
            if factor != 1.0 and production.is_planned and (production.state == 'confirmed'
                or (production.state == 'progress' and not any(wo.state in ['done', 'progress'] for wo in production.workorder_ids))):
                production._unplan_workorders()
                production._plan_workorders()
        # run scheduler for moves forecasted to not have enough in stock
        self.mo_id.filtered(lambda mo: mo.state in ['confirmed', 'progress']).move_raw_ids._trigger_scheduler()

        return {}

    def action_split_mo(self):
        for wizard in self:
            if wizard.product_qty <= 0:
                raise UserError(_("Please specify a quantity bigger than 0."))
            if wizard.product_qty >= wizard.mo_id.product_qty:
                raise UserError(_("Set a quantity that is smaller than the intial demand to split the manufacturing order."))
            new_product_qty = wizard.mo_id.product_qty - wizard.product_qty
            new_mo = wizard.mo_id._split_productions(amounts={wizard.mo_id: [wizard.product_qty, new_product_qty]}, skip_workorder_quantity=True)
            wizard.change_prod_qty()
            wizard.mo_id.message_post(body=wizard.env._("The backorder %s has been created.", new_mo[-1]._get_html_link()))
