import { registry } from "@web/core/registry";
import { markup } from "@odoo/owl";
import { ProductTemplateAccounting } from "./accounting/product_template_accounting";
import { normalize } from "@web/core/l10n/utils";

/**
 * ProductProduct, shadow of product.product in python.
 * To works properly, this model needs to be registered in the registry
 * with the key "pos_available_models". And to be instanciated with the
 * method createRelatedModels from related_models.js
 *
 * Models to load: product.product, uom.uom
 */

export class ProductTemplate extends ProductTemplateAccounting {
    static pythonModel = "product.template";

    isAllowOnlyOneLot() {
        return this.tracking === "lot" || !this.uom_id || !this.uom_id.is_pos_groupable;
    }

    isTracked() {
        const pickingType = this.models["stock.picking.type"].readAll()[0];

        return (
            ["serial", "lot"].includes(this.tracking) &&
            (pickingType.use_create_lots || pickingType.use_existing_lots)
        );
    }

    async _onScaleNotAvailable() {}

    isConfigurable() {
        return this.attribute_line_ids.find(
            (l) =>
                l.product_template_value_ids.length > 1 ||
                l.product_template_value_ids.some((v) => v.is_custom)
        );
    }

    needToConfigure() {
        return (
            this.isConfigurable() &&
            this.attribute_line_ids.length > 0 &&
            this.attribute_line_ids.some((l) => l.attribute_id.create_variant === "no_variant")
        );
    }

    isCombo() {
        return this.combo_ids.length;
    }

    get isScaleAvailable() {
        return true;
    }

    get parentCategories() {
        const categories = [];
        let category = this.categ_id;

        while (category) {
            categories.push(category.id);
            category = category.parent_id;
        }

        return categories;
    }

    get parentPosCategIds() {
        const current = [];
        const categories = this.pos_categ_ids;

        const getParent = (categ) => {
            if (categ.parent_id) {
                current.push(categ.parent_id.id);
                getParent(categ.parent_id);
            }
        };

        for (const category of categories) {
            current.push(category.id);
            getParent(category);
        }

        return current;
    }

    getImageUrl() {
        return `/web/image?model=product.template&field=image_128&id=${this.id}&unique=${this.write_date}`;
    }

    _isArchivedCombination(attributeValueIds) {
        if (!this._archived_combinations) {
            return false;
        }
        const excludedPTAV = new Set();
        let isCombinationArchived = false;
        for (const archivedCombination of this._archived_combinations) {
            const ptavCommon = archivedCombination.filter((ptav) =>
                attributeValueIds.includes(ptav)
            );
            if (ptavCommon.length === attributeValueIds.length) {
                // all attributes must be disabled from each other
                archivedCombination.forEach((ptav) => excludedPTAV.add(ptav));
            } else if (ptavCommon.length === attributeValueIds.length - 1) {
                // In this case we only need to disable the remaining ptav
                const disablePTAV = archivedCombination.find(
                    (ptav) => !attributeValueIds.includes(ptav)
                );
                excludedPTAV.add(disablePTAV);
            }
            if (ptavCommon.length === attributeValueIds.length) {
                isCombinationArchived = true;
            }
        }
        this.attribute_line_ids.forEach((attribute_line) => {
            attribute_line.product_template_value_ids.forEach((ptav) => {
                ptav["excluded"] = excludedPTAV.has(ptav.id);
            });
        });
        return isCombinationArchived;
    }

    get productDescriptionMarkup() {
        return this.public_description ? markup(this.public_description) : "";
    }

    get canBeDisplayed() {
        return this.active && this.available_in_pos;
    }

    get searchString() {
        const fields = ["name", "default_code", "barcode"];
        const raw = fields
            .map((field) => this[field] || "")
            .filter(Boolean)
            .join(" ");

        const templateContent = normalize(raw);
        const variantContent = this.product_variant_ids.map((v) => v.searchString).join(" ");

        return variantContent ? templateContent + " " + variantContent : templateContent;
    }

    get normalizedName() {
        return normalize(this.name || "");
    }

    get displayName() {
        return this.default_code
            ? `[${this.default_code}] ${this.display_name}`
            : this.display_name;
    }
}
registry.category("pos_available_models").add(ProductTemplate.pythonModel, ProductTemplate);
