# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.

from datetime import date, datetime, timedelta
from odoo.addons.hr_holidays.tests.common import TestHrHolidaysCommon
from odoo.addons.mail.tests.common import mail_new_test_user
from odoo.exceptions import ValidationError
from freezegun import freeze_time

from odoo.tests import tagged

@tagged('global_leaves')
@tagged('at_install', '-post_install')  # LEGACY at_install
class TestGlobalLeaves(TestHrHolidaysCommon):
    """ Test global leaves for a whole company, conflict resolutions """

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls.calendar_1 = cls.env['resource.calendar'].create({
            'name': 'Classic 40h/week',
            'hours_per_day': 8.0,
            'attendance_ids': [
                (0, 0, {'dayofweek': '0', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '0', 'hour_from': 13, 'hour_to': 17}),
                (0, 0, {'dayofweek': '1', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '1', 'hour_from': 13, 'hour_to': 17}),
                (0, 0, {'dayofweek': '2', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '2', 'hour_from': 13, 'hour_to': 17}),
                (0, 0, {'dayofweek': '3', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '3', 'hour_from': 13, 'hour_to': 17}),
                (0, 0, {'dayofweek': '4', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '4', 'hour_from': 13, 'hour_to': 17})
            ]
        })

        cls.calendar_2 = cls.env['resource.calendar'].create({
            'name': 'Classic 20h/week',
            'hours_per_day': 4.0,
            'attendance_ids': [
                (0, 0, {'dayofweek': '0', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '1', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '2', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '3', 'hour_from': 8, 'hour_to': 12}),
                (0, 0, {'dayofweek': '4', 'hour_from': 8, 'hour_to': 12}),
            ]
        })

        cls.global_leave = cls.env['resource.calendar.leaves'].create({
            'name': 'Global Time Off',
            'date_from': date(2022, 3, 7),
            'date_to': date(2022, 3, 7),
        })

        cls.calendar_leave = cls.env['resource.calendar.leaves'].create({
            'name': 'Global Time Off',
            'date_from': date(2022, 3, 8),
            'date_to': date(2022, 3, 8),
            'calendar_id': cls.calendar_1.id,
        })

    def test_leave_on_global_leave(self):
        with self.assertRaises(ValidationError):
            self.env['resource.calendar.leaves'].create({
                'name': 'Wrong Time Off',
                'date_from': date(2022, 3, 7),
                'date_to': date(2022, 3, 7),
                'calendar_id': self.calendar_1.id,
            })

        with self.assertRaises(ValidationError):
            self.env['resource.calendar.leaves'].create({
                'name': 'Wrong Time Off',
                'date_from': date(2022, 3, 7),
                'date_to': date(2022, 3, 7),
            })

    def test_leave_on_calendar_leave(self):
        self.env['resource.calendar.leaves'].create({
                'name': 'Correct Time Off',
                'date_from': date(2022, 3, 8),
                'date_to': date(2022, 3, 8),
                'calendar_id': self.calendar_2.id,
            })

        with self.assertRaises(ValidationError):
            self.env['resource.calendar.leaves'].create({
                'name': 'Wrong Time Off',
                'date_from': date(2022, 3, 8),
                'date_to': date(2022, 3, 8),
            })

        with self.assertRaises(ValidationError):
            self.env['resource.calendar.leaves'].create({
                'name': 'Wrong Time Off',
                'date_from': date(2022, 3, 8),
                'date_to': date(2022, 3, 8),
                'calendar_id': self.calendar_1.id,
            })

    def test_global_leave_number_of_days_with_new(self):
        """
            Check that leaves stored in memory (and not in the database)
            take into account global leaves.
        """
        global_leave = self.env['resource.calendar.leaves'].create({
            'name': 'Global Time Off',
            'date_from': datetime(2024, 1, 3, 6, 0, 0),
            'date_to': datetime(2024, 1, 3, 19, 0, 0),
            'calendar_id': self.calendar_1.id,
        })
        leave_type = self.env['hr.leave.type'].create({
            'name': 'Paid Time Off',
            'time_type': 'leave',
            'requires_allocation': False,
            'request_unit': 'day',
            'unit_of_measure': 'day',
        })
        self.employee_emp.resource_calendar_id = self.calendar_1.id

        leave = self.env['hr.leave'].create({
            'name': 'Test new leave',
            'employee_id': self.employee_emp.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': global_leave.date_from,
            'request_date_to': global_leave.date_to,
        })
        self.assertEqual(leave.number_of_days, 0, 'It is a global leave')

        leave = self.env['hr.leave'].new({
            'name': 'Test new leave',
            'employee_id': self.employee_emp.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': global_leave.date_from,
            'request_date_to': global_leave.date_to,
        })
        self.assertEqual(leave.number_of_days, 0, 'It is a global leave')

        leave = self.env['hr.leave'].new({
            'name': 'Test new leave',
            'employee_id': self.employee_emp.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': global_leave.date_from - timedelta(days=1),
            'request_date_to': global_leave.date_to + timedelta(days=1),
        })
        self.assertEqual(leave.number_of_days, 2, 'There is a global leave')

    @freeze_time('2024-12-01')
    def test_global_leave_keeps_employee_resource_leave(self):
        """
            When a global leave is created, and it happens during a leave period of an employee,
            if the employee's leave is not fully covered by the global leave, the employee's leave
            should still have resource leaves linked to it.
        """
        employee = self.employee_emp
        leave_type = self.env['hr.leave.type'].create({
            'name': 'Paid Time Off',
            'request_unit': 'hour',
            'unit_of_measure': 'hour',
            'leave_validation_type': 'both',
        })
        self.env['hr.leave.allocation'].create({
            'name': '20 days allocation',
            'holiday_status_id': leave_type.id,
            'number_of_days': 20,
            'employee_id': employee.id,
            'state': 'confirm',
            'date_from': date(2024, 12, 1),
            'date_to': date(2024, 12, 30),
        }).action_approve()

        partially_covered_leave = self.env['hr.leave'].create({
            'name': 'Holiday 1 week',
            'employee_id': employee.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': datetime(2024, 12, 3, 7, 0),
            'request_date_to': datetime(2024, 12, 5, 18, 0),
        })
        partially_covered_leave.action_approve()

        global_leave = self.env['resource.calendar.leaves'].with_user(self.env.user).create({
            'name': 'Public holiday',
            'date_from': "2024-12-04 06:00:00",
            'date_to': "2024-12-04 23:00:00",
            'calendar_id': self.calendar_1.id,
        })

        # retrieve resource leaves linked to the employee's leave
        resource_leaves = self.env['resource.calendar.leaves'].search([
            ('holiday_id', '=', partially_covered_leave.id)
        ])
        self.assertTrue(resource_leaves, 'Resource leaves linked to the employee leave should exist.')

    @freeze_time('2025-05-11')
    def test_employee_leave_with_global_leave(self):
        """
            When an employee's leave is created, if there are any public holidays within the leave period,
            the number of leave days is reduced accordingly.
            eg,.
            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            | Leave Requested  |  Leave State  | Public Holiday days  |  # days leave remains |
            |---------------------------------------------------------------------------------|
            |       5 Days     |    confirm    |        1 Days        |         4 Days        |
            |---------------------------------------------------------------------------------|
            |       4 Days     |   validate1   |        1 Days        |         3 Days        |
            |---------------------------------------------------------------------------------|
            |       3 Days     |    validate   |        1 Days        |         2 Days        |
            ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        """
        user_david = mail_new_test_user(self.env, login='david', groups='base.group_user,hr_holidays.group_hr_holidays_employee')
        user_timeoff_officer_david = mail_new_test_user(self.env, login='timeoff_officer', groups='base.group_user,hr_holidays.group_hr_holidays_employee')

        employee_david = self.env['hr.employee'].create({
            'name': 'David Employee',
            'user_id': user_david.id,
            'leave_manager_id': user_timeoff_officer_david.id,
            'parent_id': self.employee_hruser.id,
            'department_id': self.rd_dept.id,
            'resource_calendar_id': self.calendar_1.id,
        })
        leave_type = self.env['hr.leave.type'].create({
            'name': 'Sick Time Off',
            'time_type': 'leave',
            'requires_allocation': False,
            'leave_validation_type': 'both',
            'request_unit': 'day',
            'unit_of_measure': 'day',
        })

        employee_leave = self.env['hr.leave'].create({
            'name': 'Holiday 5 days',
            'employee_id': employee_david.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': datetime(2025, 5, 12),
            'request_date_to': datetime(2025, 5, 16),
        })

        self.env['resource.calendar.leaves'].with_user(self.user_hrmanager).create({
            'name': 'Public holiday day 1',
            'date_from': datetime(2025, 5, 13),
            'date_to': datetime(2025, 5, 13, 23, 59),
            'calendar_id': employee_david.resource_calendar_id.id,
        })

        self.assertEqual(employee_leave.number_of_days, 4, 'Leave duration should be reduced because of public holiday day 1')

        employee_leave.with_user(user_timeoff_officer_david).action_approve()
        self.env['resource.calendar.leaves'].with_user(self.user_hrmanager).create({
            'name': 'Public holiday day 2',
            'date_from': datetime(2025, 5, 14),
            'date_to': datetime(2025, 5, 14, 23, 59),
            'calendar_id': employee_david.resource_calendar_id.id,
        })
        self.assertEqual(employee_leave.number_of_days, 3, 'Leave duration should be reduced because of public holiday day 2')

        employee_leave.with_user(self.user_hruser).action_approve()
        self.env['resource.calendar.leaves'].with_user(self.user_hrmanager).create({
            'name': 'Public holiday day 3',
            'date_from': datetime(2025, 5, 15),
            'date_to': datetime(2025, 5, 15, 23, 59),
            'calendar_id': employee_david.resource_calendar_id.id,
        })
        self.assertEqual(employee_leave.number_of_days, 2, 'Leave duration should be reduced because of public holiday day 3')

    def test_multi_day_public_holidays_for_flexible_schedule(self):
        """
        Test that _get_unusual_days return correct value for
        multi-day holidays in flexible schedules
        """

        flex_resource = self.env['resource.resource'].create({
            'name': 'Flexible',
            'calendar_id': False,
            'hours_per_week': 40.0,
            'hours_per_day': 8,
            'tz': 'UTC',
        })

        # tuesday to thursday
        self.env['resource.calendar.leaves'].create({
            'name': '3 day holiday', 'calendar_id': False,
            'date_from': datetime(2024, 3, 5), 'date_to': datetime(2024, 3, 7, 23, 59, 59)
        })

        # monday to saturday
        start = datetime(2024, 3, 4)
        end = datetime(2024, 3, 10)

        flex_days = self.env['resource.calendar']._get_unusual_days(start, end, resource=flex_resource)

        expected = {
            '2024-03-04': False,
            '2024-03-05': True,
            '2024-03-06': True,
            '2024-03-07': True,
            '2024-03-08': False,
            '2024-03-09': False,
            '2024-03-10': False,
        }
        for day, value in expected.items():
            self.assertEqual(flex_days.get(day), value, f"Day {day} should be {'unusual' if value else 'normal'}")

    def test_public_holidays_for_consecutive_allocations(self):
        employee = self.employee_emp
        leave_type = self.env['hr.leave.type'].create({
            'name': 'Paid Time Off',
            'time_type': 'leave',
            'requires_allocation': 'yes',
        })
        self.env['hr.leave.allocation'].create([
            {
                'name': '2025 allocation',
                'holiday_status_id': leave_type.id,
                'number_of_days': 20,
                'employee_id': employee.id,
                'state': 'confirm',
                'date_from': date(2025, 1, 1),
                'date_to': date(2025, 12, 31),
            },
            {
                'name': '2026 allocation',
                'holiday_status_id': leave_type.id,
                'number_of_days': 20,
                'employee_id': employee.id,
                'state': 'confirm',
                'date_from': date(2026, 1, 1),
                'date_to': date(2026, 12, 31),
            }
        ]).action_approve()

        leave = self.env['hr.leave'].create({
            'name': 'Holiday 1 week',
            'employee_id': employee.id,
            'holiday_status_id': leave_type.id,
            'request_date_from': datetime(2025, 12, 8, 7, 0),
            'request_date_to': datetime(2026, 1, 3, 18, 0),
        })
        leave.action_approve()

        self.assertEqual(leave.number_of_days, 20, "Number of days should be 20")

        public_holiday = self.env['resource.calendar.leaves'].create({
            'name': 'Global Time Off',
            'date_from': datetime(2025, 12, 31, 23, 0, 0),
            'date_to': datetime(2026, 1, 1, 22, 59, 59),
        })

        self.assertTrue(public_holiday)
        self.assertEqual(leave.number_of_days, 19, "Number of days should be 19 as one day has been granted back to the"
                                                   "the employee for the public holiday")
