/** @odoo-module **/
import { useLayoutEffect } from "@web/owl2/utils";
import { onMounted } from "@odoo/owl";
import { CodeEditor } from "@web/core/code_editor/code_editor";
import { escapeRegExp } from "@web/core/utils/strings";

export class IrUiViewCodeEditor extends CodeEditor {
    static props = {
        ...this.props,
        invalidLocators: { type: Array, optional: true },
    };

    setup() {
        super.setup(...arguments);
        this.markers = [];

        onMounted(() => {
            this.aceEditor.getSession().on("change", () => {
                // Markers have fixed pixel positions, so they get wonky on change.
                this.clearMarkers();
            });
        });

        useLayoutEffect(
            (arch, invalid_locators) => {
                if (arch && invalid_locators) {
                    this.highlightInvalidLocators(arch, invalid_locators);
                    return () => this.clearMarkers();
                }
            },
            () => [this.props.value, this.props.invalidLocators]
        );
    }

    async highlightInvalidLocators(arch, invalid_locators) {
        const { doc } = this.aceEditor.session;
        for (const spec of invalid_locators) {
            if (spec.broken_hierarchy) {
                continue;
            }
            const { tag, attrib, sourceline } = spec;
            const attribRegex = Object.entries(attrib)
                .map(([key, value]) => {
                    const escapedValue = escapeRegExp(value).replace(/"/g, '("|&quot;)');
                    return (
                        `(?=[^>]*?\\b${escapeRegExp(key)}\\s*=\\s*` +
                        `(?:"[^"]*${escapedValue}[^"]*"|'[^']*${escapedValue}[^']*'))`
                    );
                })
                .join("");
            const nodeRegex = new RegExp(`<${escapeRegExp(tag)}\\s+${attribRegex}[^>]*>`, "g");
            for (const match of arch.matchAll(nodeRegex)) {
                const startIndex = match.index;
                const endIndex = startIndex + match[0].length;
                const startPos = doc.indexToPosition(startIndex);
                const endPos = doc.indexToPosition(endIndex);
                if (startPos.row + 1 === sourceline) {
                    const range = new window.ace.Range(
                        startPos.row,
                        startPos.column,
                        endPos.row,
                        endPos.column
                    );
                    this.markers.push(
                        this.aceEditor.session.addMarker(range, "invalid_locator", "text")
                    );
                }
            }
        }
    }

    clearMarkers() {
        this.markers.forEach((marker) => this.aceEditor.session.removeMarker(marker));
        this.markers = [];
    }
}
