# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import api, fields, models
from odoo.tools import format_amount
from odoo.tools.sql import column_exists, create_column


class SaleOrderLine(models.Model):
    _inherit = "sale.order.line"

    _name_search_services_index = models.Index("(order_id DESC, sequence, id) WHERE is_service IS TRUE")

    # used to know if generate a task and/or a project, depending on the product settings
    is_service = fields.Boolean("Is a Service", compute='_compute_is_service', store=True, compute_sudo=True, export_string_translation=False)

    def _domain_sale_line_service(self, **kwargs):
        """
        Get the default generic services domain for sale.order.line.
        You can filter out domain leafs by passing kwargs of the form 'check_<leaf_field>=False'.
        Only 'is_service' cannot be disabled.

        :param kwargs: boolean kwargs of the form 'check_<leaf_field>=False'
        :return: a valid domain
        """
        domain = [('is_service', '=', True)]
        if kwargs.get("check_is_expense", True):
            domain.append(('is_expense', '=', False))
        if kwargs.get("check_state", True):
            domain.append(('state', '=', 'sale'))
        return domain

    @api.depends('product_id.type')
    def _compute_is_service(self):
        self.fetch(['is_service', 'product_id'])
        self.product_id.fetch(['type'])
        for so_line in self:
            so_line.is_service = so_line.product_id.type == 'service'

    @api.depends('is_service')
    @api.depends_context('formatted_display_name')
    def _compute_display_name(self):
        sol = self.browse()
        if self.env.context.get('formatted_display_name'):
            for line in self:
                if line.is_service:
                    name = f"{line.order_id.name} - {line.product_id.display_name}"
                    formatted_price = format_amount(self.env, line.product_id.lst_price, line.currency_id)
                    line.display_name = f"{name} --({line.order_partner_id.name})-- --({formatted_price})--"
                    sol |= line
        super(SaleOrderLine, self - sol)._compute_display_name()

    def _auto_init(self):
        """
        Create column to stop ORM from computing it himself (too slow)
        """
        if not column_exists(self.env.cr, 'sale_order_line', 'is_service'):
            create_column(self.env.cr, 'sale_order_line', 'is_service', 'bool')
            self.env.cr.execute("""
                UPDATE sale_order_line line
                SET is_service = (pt.type = 'service')
                FROM product_product pp
                LEFT JOIN product_template pt ON pt.id = pp.product_tmpl_id
                WHERE pp.id = line.product_id
            """)
        return super()._auto_init()

    def _additional_name_per_id(self):
        name_per_id = super()._additional_name_per_id() if not self.env.context.get('hide_partner_ref') else {}
        if not self.env.context.get('with_price_unit'):
            return name_per_id

        for sols in self.grouped(lambda sol: (sol.order_id, sol.product_id)).values():
            if len(sols) <= 1 or not all(sol.is_service for sol in sols):
                continue
            for line in sols:
                additional_name = name_per_id.get(line.id)
                name = format_amount(self.env, line.price_unit, line.currency_id)
                if additional_name:
                    name += f' {additional_name}'
                name_per_id[line.id] = f'- {name}'

        return name_per_id

    @api.model
    def name_search(self, name='', domain=None, operator='ilike', limit=100):
        domain = domain or []
        # optimization for a SOL services name_search, to avoid joining on sale_order with too many lines
        if domain and ('is_service', '=', True) in domain and operator in ('like', 'ilike') and limit is not None:
            sols = self.search_fetch(
                domain, ['display_name'], limit=limit, order='order_id.id DESC, sequence, id',
            )
            return [(sol.id, sol.display_name) for sol in sols]
        return super().name_search(name, domain, operator, limit)
