# Part of Odoo. See LICENSE file for full copyright and licensing details.

from dateutil.relativedelta import relativedelta

from odoo import fields, models, api, _
from odoo.exceptions import UserError

from zoneinfo import ZoneInfo
from datetime import timezone


class HrLeave(models.Model):
    _inherit = 'hr.leave'

    l10n_fr_date_to_changed = fields.Boolean(export_string_translation=False)

    def _l10n_fr_leave_applies(self):
        # The french l10n is meant to be computed only in very specific cases:
        # - there is only one employee affected by the leave
        # - the company is french
        # - the leave_type is the reference leave_type of that company
        self.ensure_one()
        return self.employee_id and \
               self.company_id.country_id.code == 'FR' and \
               self.resource_calendar_id != self.company_id.resource_calendar_id and \
               self.holiday_status_id == self.company_id._get_fr_reference_leave_type()

    def _get_fr_date_from_to(self, date_from, date_to):
        self.ensure_one()
        # What we need to compute is how much we will need to push date_to in order to account for the lost days

        # The following computation doesn't work for resource calendars in
        # which the employee works zero hours.
        if not (self.resource_calendar_id.attendance_ids):
            raise UserError(_("An employee can't take paid time off in a period without any work hours."))

        if self.leave_type_request_unit != 'hour':
            # Use company's working schedule hours for the leave to avoid duration calculation issues.
            def adjust_date_range(date_from, date_to, from_period, to_period, attendance_ids, employee_id):
                period_ids_from = attendance_ids.filtered(lambda a: a.day_period in from_period
                                                                    and int(a.dayofweek) == date_from.weekday())
                period_ids_to = attendance_ids.filtered(lambda a: a.day_period in to_period
                                                                    and int(a.dayofweek) == date_to.weekday())
                if period_ids_from:
                    min_hour = min(attendance.hour_from for attendance in period_ids_from)
                    date_from = self._to_utc(date_from, min_hour, employee_id)
                if period_ids_to:
                    max_hour = max(attendance.hour_to for attendance in period_ids_to)
                    date_to = self._to_utc(date_to, max_hour, employee_id)
                return date_from, date_to

            if self.leave_type_request_unit == 'half_day':
                from_period = ['morning'] if self.request_date_from_period == 'am' else ['afternoon']
                to_period = ['morning'] if self.request_date_to_period == 'am' else ['afternoon']
            else:
                from_period = ['morning', 'afternoon']
                to_period = ['morning', 'afternoon']
            attendance_ids = self.company_id.resource_calendar_id.attendance_ids | self.resource_calendar_id.attendance_ids
            date_from, date_to = adjust_date_range(date_from, date_to, from_period, to_period, attendance_ids, self.employee_id)

        similar = date_from.date() == date_to.date() and self.request_date_from_period == self.request_date_to_period
        if self.leave_type_request_unit == 'half_day' and similar and self.request_date_from_period == 'am':
            # In normal workflows leave_type_request_unit = 'half_day' implies that date_from and date_to are the same
            # leave_type_request_unit = 'half_day' allows us to choose between `am` and `pm`
            # In a case where we work from mon-wed and request a half day in the morning
            # we do not want to push date_to since the next work attendance is actually in the afternoon
            date_from_dayofweek = str(date_from.weekday())
            # Fetch the attendances we care about
            attendance_ids = self.resource_calendar_id.attendance_ids.filtered(lambda a:
                a.dayofweek == date_from_dayofweek
            )
            if len(attendance_ids) == 2:
                # The employee took the morning off on a day where he works the afternoon aswell
                return (date_from, date_to)

        # Check calendars for working days until we find the right target, start at date_to + 1 day
        # Postpone date_target until the next working day
        date_start = date_from
        date_target = date_to
        # It is necessary to move the start date up to the first work day of
        # the employee calendar as otherwise days worked on by the company
        # calendar before the actual start of the leave would be taken into
        # account.
        while not self.resource_calendar_id._works_on_date(date_start):
            date_start += relativedelta(days=1)
        while not self.resource_calendar_id._works_on_date(date_target + relativedelta(days=1)):
            date_target += relativedelta(days=1)

        # Undo the last day increment
        return (date_start, date_target)

    @api.depends('request_date_from_period', 'request_date_to_period', 'request_hour_from', 'request_hour_to',
                'request_date_from', 'request_date_to', 'leave_type_request_unit', 'employee_id')
    def _compute_date_from_to(self):
        super()._compute_date_from_to()
        for leave in self:
            if leave._l10n_fr_leave_applies():
                new_date_from, new_date_to = leave._get_fr_date_from_to(leave.date_from, leave.date_to)
                if new_date_from != leave.date_from:
                    leave.date_from = new_date_from
                if new_date_to != leave.date_to:
                    leave.date_to = new_date_to
                    leave.l10n_fr_date_to_changed = True
                else:
                    leave.l10n_fr_date_to_changed = False

    def _get_durations(self, check_leave_type=True, resource_calendar=None, additional_domain=[]):
        """
        In french time off laws, if an employee has a part time contract, when taking time off
        before one of his off day (compared to the company's calendar) it should also count the time
        between the time off and the next calendar work day/company off day (weekends).

        For example take an employee working mon-wed in a company where the regular calendar is mon-fri.
        If the employee were to take a time off ending on wednesday, the legal duration would count until friday.
        """
        if not resource_calendar:
            fr_leaves = self.filtered(lambda leave: leave._l10n_fr_leave_applies())
            duration_by_leave_id = super(HrLeave, self - fr_leaves)._get_durations(resource_calendar=resource_calendar)
            fr_leaves_by_company = fr_leaves.grouped('company_id')
            if fr_leaves:
                public_holidays = self.env['resource.calendar.leaves'].search([
                    ('resource_id', '=', False),
                    ('company_id', 'in', fr_leaves.company_id.ids + [False]),
                    ('date_from', '<', max(fr_leaves.mapped('date_to')) + relativedelta(days=1)),
                    ('date_to', '>', min(fr_leaves.mapped('date_from')) - relativedelta(days=1)),
                ])
            for company, leaves in fr_leaves_by_company.items():
                company_cal = company.resource_calendar_id
                holidays_days_list = []
                public_holidays_filtered = public_holidays.filtered_domain([
                    ('calendar_id', 'in', [False, company_cal.id]),
                    ('company_id', '=', company.id)
                ])
                for holiday in public_holidays_filtered:
                    tz = ZoneInfo(holiday.write_uid.tz or 'UTC')
                    current = holiday.date_from.replace(tzinfo=timezone.utc).astimezone(tz).date()
                    holiday_date_to = holiday.date_to.replace(tzinfo=timezone.utc).astimezone(tz).date()
                    while current <= holiday_date_to:
                        holidays_days_list.append(current)
                        current += relativedelta(days=1)
                for leave in leaves:
                    if leave.leave_type_request_unit == 'half_day':
                        duration_by_leave_id.update(leave._get_durations(resource_calendar=company_cal))
                        continue
                    # Extend the end date to next working day
                    date_start = leave.date_from
                    date_end = leave.date_to
                    while not leave.resource_calendar_id._works_on_date(date_start):
                        date_start += relativedelta(days=1)
                    extended_date_end = date_end
                    while not company_cal._works_on_date(extended_date_end + relativedelta(days=1)):
                        extended_date_end += relativedelta(days=1)
                    # Count number of days in company calendar
                    current = date_start.date()
                    end_date = extended_date_end.date()
                    legal_days = 0.0
                    while current <= end_date:
                        if current in holidays_days_list:
                            current += relativedelta(days=1)
                            continue
                        if company_cal._works_on_date(current):
                            legal_days += 1.0
                        current += relativedelta(days=1)
                    standard_duration = super()._get_durations(resource_calendar=resource_calendar)
                    _, hours = standard_duration.get(leave.id, (0.0, 0.0))

                    duration_by_leave_id[leave.id] = (legal_days, hours)

            return duration_by_leave_id
        return super()._get_durations(check_leave_type, resource_calendar, additional_domain)
